#include <stdexcept>
#include "File.h"
#include "AbstractMailMessageReader.h"
#include "FileMailMessageReader.h"
#include "MaildirMailMessageReader.h"
#include "StreamMailMessageReader.h"
#include "MailMessageReaderFactory.h"

MailMessageReaderFactory::MailMessageReaderFactory()
  : m_ignoreFrom(false),
    m_ignoreContentLength(true)
{
}

// Any Maildir directory is valid.
// Just about any file is acceptable except for dot files inside
// of a Maildir directory.  The latter restriction prevents us from
// processing password files, IMAPD cache files, etc.
bool MailMessageReaderFactory::isAcceptableFile(const File &file)
{
  if (file.isSelfOrParent()) {
    return false;
  } else if (file.isFile()) {
    return !(file.isHidden() && MaildirMailMessageReader::isMaildir(file.parent()));
  } else {
    return MaildirMailMessageReader::isMaildir(file);
  }
}

OWNED AbstractMailMessageReader *MailMessageReaderFactory::createReader(const File &file)
{
  if (file.isDirectory()) {
    if (!MaildirMailMessageReader::isMaildir(file)) {
      throw runtime_error(string("not a maildir: ") + file.getPath());
    }
    return new MaildirMailMessageReader(file);
  } else {
    Ptr<FileMailMessageReader> reader(new FileMailMessageReader);
    reader->open(file.getPath(), m_ignoreFrom, m_ignoreContentLength);
    return reader.release();
  }
}

OWNED AbstractMailMessageReader *MailMessageReaderFactory::createStdinReader()
{
  Ptr<StreamMailMessageReader> reader(new StreamMailMessageReader);
  reader->setStream(&cin, false, m_ignoreFrom, m_ignoreContentLength);
  return reader.release();
}
