-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

-- Grammar:
-- Node passed in is the identifier node of the generic unit being instantiated
-- The node to the right of this is the generic_actual_part if it exists.
-- Thus:
--   ... ... ... identifier [generic_actual_part]
--                                    |
--                             name_argument_list
--
--
--  name_argument_list :
--        named_argument_association
--      | positional_argument_association ;
--
--  named_argument_association :
--        named_argument_association simple_name expression
--      | simple_name expression ;
--
--  positional_argument_association :
--        positional_argument_association expression
--      | expression ;
--
-- Wffs:
-- 1. If generic has formal parameters then actual part must exist.
-- 2. If generic has not formals then actual part must not exist.
-- 3. There must be one actual for each formal.
-- 4. Each actual must be of the right type and kind (i.e. object or type).
--
-- Actions:
-- Wellformed formal/actual associations are written to the Dictionary
-- and associated with the Instantiation_Sym.

separate (Sem.CompUnit)
procedure Wf_Generic_Actual_Part
  (Generic_Node      : in     STree.SyntaxNode;
   Generic_Sym       : in     Dictionary.Symbol;
   Instantiation_Sym : in     Dictionary.Symbol;
   Scope             : in     Dictionary.Scopes;
   Error_Found       :    out Boolean) is
   Actual_Part_Node : STree.SyntaxNode;

   procedure Check_Missing_Actual_Part
     (Generic_Node     : in     STree.SyntaxNode;
      Actual_Part_Node : in     STree.SyntaxNode;
      Generic_Sym      : in     Dictionary.Symbol;
      Error_Found      : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Actual_Part_Node,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Generic_Node,
   --#                                         Generic_Sym,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Error_Found                from *,
   --#                                         Actual_Part_Node,
   --#                                         Dictionary.Dict,
   --#                                         Generic_Sym;
   --# pre Syntax_Node_Type (Generic_Node, STree.Table) = SP_Symbols.identifier and
   --#   (Syntax_Node_Type (Actual_Part_Node, STree.Table) = SP_Symbols.generic_actual_part or
   --#      Actual_Part_Node = STree.NullNode);
   is
   begin
      if Dictionary.GetNumberOfGenericFormalParameters (Generic_Sym) > 0 and then Actual_Part_Node = STree.NullNode then
         ErrorHandler.Semantic_Error
           (Err_Num   => 633,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Generic_Node),
            Id_Str    => Node_Lex_String (Node => Generic_Node));
         Error_Found := True;
      end if;
   end Check_Missing_Actual_Part;

   ---------------------------

   procedure Check_Superfluous_Actual_Part
     (Generic_Node     : in     STree.SyntaxNode;
      Actual_Part_Node : in     STree.SyntaxNode;
      Generic_Sym      : in     Dictionary.Symbol;
      Error_Found      : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Actual_Part_Node,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Generic_Node,
   --#                                         Generic_Sym,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Error_Found                from *,
   --#                                         Actual_Part_Node,
   --#                                         Dictionary.Dict,
   --#                                         Generic_Sym;
   --# pre Syntax_Node_Type (Generic_Node, STree.Table) = SP_Symbols.identifier and
   --#   (Syntax_Node_Type (Actual_Part_Node, STree.Table) = SP_Symbols.generic_actual_part or
   --#      Actual_Part_Node = STree.NullNode);
   is
   begin
      if Dictionary.GetNumberOfGenericFormalParameters (Generic_Sym) = 0 and then Actual_Part_Node /= STree.NullNode then
         ErrorHandler.Semantic_Error
           (Err_Num   => 634,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Actual_Part_Node),
            Id_Str    => Node_Lex_String (Node => Generic_Node));
         Error_Found := True;
      end if;
   end Check_Superfluous_Actual_Part;

   ---------------------------

   procedure Process_Generic_Type_Parameter
     (Formal_Sym        : in     Dictionary.Symbol;
      Actual_Expression : in     STree.SyntaxNode;
      Instantiation_Sym : in     Dictionary.Symbol;
      Scope             : in     Dictionary.Scopes;
      Error_Found       : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         Actual_Expression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict,
   --#         Error_Found                from *,
   --#                                         Actual_Expression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Formal_Sym,
   --#                                         GlobalComponentData,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Actual_Expression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Formal_Sym,
   --#                                         GlobalComponentData,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State                  from *,
   --#                                         Actual_Expression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Actual_Expression, STree.Table) = SP_Symbols.expression;
   --# post STree.Table = STree.Table~;
   is
      Expression_Result : Exp_Record;
      Is_A_Name         : Boolean;
      Unused_Ref_Vars   : SeqAlgebra.Seq;
      Actual_Type       : Dictionary.Symbol;

      procedure Check_Type_Compatibility
        (Error_Position    : in     LexTokenManager.Token_Position;
         Formal_Type       : in     Dictionary.Symbol;
         Instantiation_Sym : in     Dictionary.Symbol;
         Scope             : in     Dictionary.Scopes;
         Actual_Type       : in out Dictionary.Symbol;
         Error_Found       : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives Actual_Type,
      --#         Error_Found                from *,
      --#                                         Actual_Type,
      --#                                         Dictionary.Dict,
      --#                                         Formal_Type,
      --#                                         Instantiation_Sym,
      --#                                         Scope &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Actual_Type,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Error_Position,
      --#                                         Formal_Type,
      --#                                         Instantiation_Sym,
      --#                                         LexTokenManager.State,
      --#                                         Scope,
      --#                                         SPARK_IO.File_Sys;
      is

         Formal_Element_Or_Index : Dictionary.Symbol;
         Actual_Element_Or_Index : Dictionary.Symbol;
         Formal_Index_Iterator   : Dictionary.Iterator;
         Actual_Index_Iterator   : Dictionary.Iterator;

         function Is_Valid_Association
           (Formal_Type       : Dictionary.Symbol;
            Actual_Type       : Dictionary.Symbol;
            Instantiation_Sym : Dictionary.Symbol;
            Scope             : Dictionary.Scopes)
           return              Boolean
         --# global in Dictionary.Dict;
         is
            Result                            : Boolean;
            Previously_Associated_Actual_Type : Dictionary.Symbol;
            Formal_Type_Local                 : Dictionary.Symbol;
         begin
            -- If formal type has already been associated (i.e. earlier in the parameter list), then
            -- get the previously given actual before doing the rest of the checks
            Previously_Associated_Actual_Type := Dictionary.ActualOfGenericFormal (Formal_Type, Instantiation_Sym);
            if Previously_Associated_Actual_Type /= Dictionary.NullSymbol then
               -- there is an earlier association for us to use
               Formal_Type_Local := Previously_Associated_Actual_Type;
            else
               -- no previous association so just run check on the formal type itself
               Formal_Type_Local := Formal_Type;
            end if;

            --PNA--14/12/05 as a result of above addition, case 2 below may be no longer needed

            -- Now go on to check the element and indexes thus:
            --   1. If the formal is a generic type then the actual must be a valid association
            --   2. If the formal is a generic formal type parameter then the actual must be the
            --      actual associated with that type earlier in the checking of this actual part
            --   3. If the formal is a predefined Ada type then the actual must identical.
            if Dictionary.TypeIsGeneric (Formal_Type_Local) then
               --PNA-- and then not Dictionary.TypeIsGenericArray (Formal_Type_Local) then
               -- Case 1.  Primitive generic types
               Result := Dictionary.IsValidGenericTypeAssociation (Formal_Type_Local, Actual_Type, Scope);
               --PNA-- elsif Dictionary.TypeIsGenericArray (Formal_Type_Local) then
               --PNA--    -- Case 2.  We are processing an array of arrays.
               --PNA--    Result := Actual_Type = Dictionary.ActualOfGenericFormal (Formal_Type_Local,
               --PNA--                                                              Instantiation_Sym);
            else -- straight Ada type
               Result := Actual_Type = Formal_Type_Local;
            end if;
            return Result;
         end Is_Valid_Association;

      begin -- Check_Type_Compatibility
         if Dictionary.TypeIsGenericArray (Formal_Type) then
            if Dictionary.TypeIsArray (Actual_Type) then
               -- Both are arrays so check elements and indexes for compatibility
               -- First check element type
               Formal_Element_Or_Index := Dictionary.GetArrayComponent (Formal_Type);
               Actual_Element_Or_Index := Dictionary.GetArrayComponent (Actual_Type);

               if not Is_Valid_Association
                 (Formal_Type       => Formal_Element_Or_Index,
                  Actual_Type       => Actual_Element_Or_Index,
                  Instantiation_Sym => Instantiation_Sym,
                  Scope             => Scope) then
                  ErrorHandler.Semantic_Error2
                    (Err_Num   => 645,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Error_Position,
                     Id_Str1   => Dictionary.GetSimpleName (Actual_Element_Or_Index),
                     Id_Str2   => Dictionary.GetSimpleName (Formal_Element_Or_Index));
                  Error_Found := True;
                  Actual_Type := Dictionary.GetUnknownTypeMark;
               end if;
               -- now loop through indexes
               Formal_Index_Iterator := Dictionary.FirstArrayIndex (Formal_Type);
               Actual_Index_Iterator := Dictionary.FirstArrayIndex (Actual_Type);
               loop
                  exit when (Formal_Index_Iterator = Dictionary.NullIterator)
                    and then (Actual_Index_Iterator = Dictionary.NullIterator); -- normal exit

                  if Formal_Index_Iterator = Dictionary.NullIterator then
                     -- Formal has finished first
                     Error_Found := True;
                     Actual_Type := Dictionary.GetUnknownTypeMark;
                     ErrorHandler.Semantic_Error2
                       (Err_Num   => 647,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Error_Position,
                        Id_Str1   => Dictionary.GetSimpleName (Actual_Type),
                        Id_Str2   => Dictionary.GetSimpleName (Formal_Type));
                     exit;
                  end if;

                  if Actual_Index_Iterator = Dictionary.NullIterator then
                     -- Actual has finished first
                     Error_Found := True;
                     Actual_Type := Dictionary.GetUnknownTypeMark;
                     ErrorHandler.Semantic_Error2
                       (Err_Num   => 648,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Error_Position,
                        Id_Str1   => Dictionary.GetSimpleName (Actual_Type),
                        Id_Str2   => Dictionary.GetSimpleName (Formal_Type));
                     exit;
                  end if;

                  -- neither finished, so check they are compatible
                  Formal_Element_Or_Index := Dictionary.CurrentSymbol (Formal_Index_Iterator);
                  Actual_Element_Or_Index := Dictionary.CurrentSymbol (Actual_Index_Iterator);
                  if not Is_Valid_Association
                    (Formal_Type       => Formal_Element_Or_Index,
                     Actual_Type       => Actual_Element_Or_Index,
                     Instantiation_Sym => Instantiation_Sym,
                     Scope             => Scope) then
                     ErrorHandler.Semantic_Error2
                       (Err_Num   => 646,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Error_Position,
                        Id_Str1   => Dictionary.GetSimpleName (Actual_Element_Or_Index),
                        Id_Str2   => Dictionary.GetSimpleName (Formal_Element_Or_Index));
                     Error_Found := True;
                     Actual_Type := Dictionary.GetUnknownTypeMark;
                  end if;

                  Formal_Index_Iterator := Dictionary.NextSymbol (Formal_Index_Iterator);
                  Actual_Index_Iterator := Dictionary.NextSymbol (Actual_Index_Iterator);
               end loop;

               -- Both formal and actual must be same constrained/unconstrained
               if Dictionary.IsUnconstrainedArrayType (Formal_Type)
                 and then not Dictionary.IsUnconstrainedArrayType (Actual_Type) then
                  Error_Found := True;
                  Actual_Type := Dictionary.GetUnknownTypeMark;
                  ErrorHandler.Semantic_Error2
                    (Err_Num   => 649,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Error_Position,
                     Id_Str1   => Dictionary.GetSimpleName (Actual_Type),
                     Id_Str2   => Dictionary.GetSimpleName (Formal_Type));
               end if;
               -- reverse check
               if Dictionary.IsUnconstrainedArrayType (Actual_Type)
                 and then not Dictionary.IsUnconstrainedArrayType (Formal_Type) then
                  Error_Found := True;
                  Actual_Type := Dictionary.GetUnknownTypeMark;
                  ErrorHandler.Semantic_Error2
                    (Err_Num   => 650,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Error_Position,
                     Id_Str1   => Dictionary.GetSimpleName (Actual_Type),
                     Id_Str2   => Dictionary.GetSimpleName (Formal_Type));
               end if;
            else
               -- Formal is an array but actual isn't, so definitely wrong
               ErrorHandler.Semantic_Error2
                 (Err_Num   => 636,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Error_Position,
                  Id_Str1   => Dictionary.GetSimpleName (Actual_Type),
                  Id_Str2   => Dictionary.GetSimpleName (Formal_Type));
               Error_Found := True;
               Actual_Type := Dictionary.GetUnknownTypeMark;
            end if;
         else -- Formal is not an array so just check actual against formal
            if not Dictionary.IsValidGenericTypeAssociation (Formal_Type, Actual_Type, Scope) then
               ErrorHandler.Semantic_Error2
                 (Err_Num   => 636,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Error_Position,
                  Id_Str1   => Dictionary.GetSimpleName (Actual_Type),
                  Id_Str2   => Dictionary.GetSimpleName (Formal_Type));
               Error_Found := True;
               Actual_Type := Dictionary.GetUnknownTypeMark;
            end if;
         end if;
      end Check_Type_Compatibility;

   begin -- Process_Generic_Type_Parameter

      -- The actual parameter takes the form of an expression but must actually be a typemark.
      -- We can use WalkName to recover the symbol, then check that it is a type and that it
      -- compatible with generic type.  Matching pairs are added to the dictionary and associated
      -- with the symbol of the instantiated unit.
      SeqAlgebra.CreateSeq (TheHeap, Unused_Ref_Vars);
      Walk_Name
        (Exp_Node       => Actual_Expression,
         Scope          => Scope,
         Component_Data => GlobalComponentData,
         Result         => Expression_Result,
         Is_A_Name      => Is_A_Name,
         Ref_Var_Param  => Unused_Ref_Vars);
      SeqAlgebra.DisposeOfSeq (TheHeap, Unused_Ref_Vars);
      if not Is_A_Name then
         -- we have a general expression which cannot possibly be a type mark
         Error_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 95,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Actual_Expression),
            Id_Str    => LexTokenManager.Null_String);
      elsif Expression_Result.Sort /= Is_Type_Mark then
         -- we have a name but it's not a type mark
         Error_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 95,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Actual_Expression),
            Id_Str    => LexTokenManager.Null_String);
      else
         -- we have a type mark
         Actual_Type := Expression_Result.Type_Symbol;
         Check_Type_Compatibility
           (Error_Position    => Node_Position (Node => Actual_Expression),
            Formal_Type       => Dictionary.GetType (Formal_Sym),
            Instantiation_Sym => Instantiation_Sym,
            Scope             => Scope,
            Actual_Type       => Actual_Type,
            Error_Found       => Error_Found);
         -- above call will change Actual_Type to the UnknownType if there is an error
         Dictionary.AddGenericAssociation
           (SubprogramOrPackage => Instantiation_Sym,
            Comp_Unit           => ContextManager.Ops.Current_Unit,
            Declaration         => Dictionary.Location'(Start_Position => Node_Position (Node => Actual_Expression),
                                                        End_Position   => Node_Position (Node => Actual_Expression)),
            FormalSym           => Dictionary.GetType (Formal_Sym),
            ActualSym           => Actual_Type);
      end if;
   end Process_Generic_Type_Parameter;

   ---------------------------

   procedure Process_Generic_Object_Parameter
     (Formal_Sym        : in     Dictionary.Symbol;
      Actual_Expression : in     STree.SyntaxNode;
      Instantiation_Sym : in     Dictionary.Symbol;
      Scope             : in     Dictionary.Scopes;
      Error_Found       : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out Aggregate_Stack.State;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from Actual_Expression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Formal_Sym,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from Actual_Expression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Formal_Sym,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Error_Found,
   --#         Statistics.TableUsage      from *,
   --#                                         Actual_Expression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Formal_Sym,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Actual_Expression, STree.Table) = SP_Symbols.expression;
   --# post STree.Table = STree.Table~;
   is
      Actual_Type           : Dictionary.Symbol;
      Unused_Component_Data : ComponentManager.ComponentData;
      Unwanted_Seq          : SeqAlgebra.Seq;
      Exp_Result            : Exp_Record;
      Scalar_Store_Val      : LexTokenManager.Lex_String;
      Constant_Name         : LexTokenManager.Lex_String;
      Constant_Sym          : Dictionary.Symbol;
      Constant_Location     : Dictionary.Location;
   begin
      -- genericTBD -- do we allow formal objects that are arrays?

      Actual_Type := Dictionary.GetType (Formal_Sym);
      -- The Actual_Type we have got here might be a predefined Ada type such as Integer.
      -- If so, then that is the type we expect the actual expression to be; however,
      -- if the type is a generic type then we need to look up the actual associated with it:
      if Dictionary.TypeIsGeneric (Actual_Type) then
         Actual_Type := Dictionary.ActualOfGenericFormal (Actual_Type, Instantiation_Sym);
      end if;

      -- walk expression
      SeqAlgebra.CreateSeq (TheHeap, Unwanted_Seq);
      ComponentManager.Initialise (Unused_Component_Data);
      --# accept Flow, 10, Unused_Component_Data, "Expected ineffective assignment";
      Walk_Expression_P.Walk_Expression
        (Exp_Node                => Actual_Expression,
         Scope                   => Scope,
         Type_Context            => Actual_Type,
         Context_Requires_Static => False,
         Ref_Var                 => Unwanted_Seq,
         Result                  => Exp_Result,
         Component_Data          => Unused_Component_Data,
         The_Heap                => TheHeap);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, Unwanted_Seq);
      Maths.StorageRep (Exp_Result.Value, Scalar_Store_Val); -- scalar value if needed later

      -- check constant
      if Exp_Result.Is_Constant then
         Assignment_Check
           (Position    => Node_Position (Node => Actual_Expression),
            Scope       => Scope,
            Target_Type => Actual_Type,
            Exp_Result  => Exp_Result);
         if Exp_Result = Unknown_Type_Record then
            Error_Found := True;
         else
            -- Fundamentally ok to add
            -- mark any errors (both to avoid use of invalid instantiation and to supress rule generation)
            if Exp_Result.Errors_In_Expression then
               Error_Found := True;
            end if;

            -- first add a constant declaration to local scope of Instantiation_Sym
            -- synthetic constant can have same name as formal it is associated with
            Constant_Name := Dictionary.GetSimpleName (Formal_Sym);
            -- and can be "located" at expression node
            Constant_Location :=
              Dictionary.Location'
              (Start_Position => Node_Position (Node => Actual_Expression),
               End_Position   => Node_Position (Node => Actual_Expression));

            if Dictionary.IsScalarTypeMark (Actual_Type, Scope) then
               Dictionary.AddScalarConstant
                 (Name            => Constant_Name,
                  TypeMark        => Actual_Type,
                  TypeReference   => Constant_Location,
                  Value           => Scalar_Store_Val,
                  ExpIsWellFormed => not Exp_Result.Errors_In_Expression,
                  ExpNode         => STree.NodeToRef (Actual_Expression),
                  Static          => Exp_Result.Is_Static,
                  Comp_Unit       => ContextManager.Ops.Current_Unit,
                  Declaration     => Constant_Location,
                  Scope           => Dictionary.LocalScope (Instantiation_Sym),
                  Context         => Dictionary.ProofContext);
            elsif Dictionary.IsArrayTypeMark (Actual_Type, Scope) then
               Dictionary.AddArrayConstant
                 (Name            => Constant_Name,
                  TypeMark        => Actual_Type,
                  TypeReference   => Constant_Location,
                  Value           => LexTokenManager.Null_String,
                  ExpIsWellFormed => not Exp_Result.Errors_In_Expression,
                  ExpNode         => STree.NodeToRef (Actual_Expression),
                  Static          => Exp_Result.Is_Static,
                  Comp_Unit       => ContextManager.Ops.Current_Unit,
                  Declaration     => Constant_Location,
                  Scope           => Dictionary.LocalScope (Instantiation_Sym),
                  Context         => Dictionary.ProofContext);
            else -- must be record
               Dictionary.AddRecordConstant
                 (Name            => Constant_Name,
                  TheType         => Actual_Type,
                  TypeReference   => Constant_Location,
                  ExpIsWellFormed => not Exp_Result.Errors_In_Expression,
                  ExpNode         => STree.NodeToRef (Actual_Expression),
                  Comp_Unit       => ContextManager.Ops.Current_Unit,
                  Declaration     => Constant_Location,
                  Scope           => Dictionary.LocalScope (Instantiation_Sym),
                  Context         => Dictionary.ProofContext);
            end if;

            -- have to get constant symbol back via lookup
            Constant_Sym :=
              Dictionary.LookupItem
              (Name              => Constant_Name,
               Scope             => Dictionary.LocalScope (Instantiation_Sym),
               Context           => Dictionary.ProofContext,
               Full_Package_Name => False);

            -- it must be there!
            SystemErrors.RT_Assert
              (C       => Constant_Sym /= Dictionary.NullSymbol,
               Sys_Err => SystemErrors.Assertion_Failure,
               Msg     => "constant added at subprogram instantiation not found in Wf_Generic_Actual_Part");

            Dictionary.AddGenericAssociation
              (SubprogramOrPackage => Instantiation_Sym,
               Comp_Unit           => ContextManager.Ops.Current_Unit,
               Declaration         => Dictionary.Location'(Start_Position => Node_Position (Node => Actual_Expression),
                                                           End_Position   => Node_Position (Node => Actual_Expression)),
               FormalSym           => Formal_Sym,
               ActualSym           => Constant_Sym);
         end if;
      else
         Error_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 640,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Actual_Expression),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end Process_Generic_Object_Parameter;

   ---------------------------

   procedure Process_Named_Argument_Association
     (Nam_Arg_Ass_Node  : in     STree.SyntaxNode;
      Generic_Sym       : in     Dictionary.Symbol;
      Instantiation_Sym : in     Dictionary.Symbol;
      Scope             : in     Dictionary.Scopes;
      Error_Found       : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         Error_Found,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Generic_Sym,
   --#                                         GlobalComponentData,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Nam_Arg_Ass_Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Generic_Sym,
   --#                                         GlobalComponentData,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Nam_Arg_Ass_Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Nam_Arg_Ass_Node, STree.Table) = SP_Symbols.named_argument_association;
   --# post STree.Table = STree.Table~;
   is
      Formal_It       : Dictionary.Iterator;
      Current_Formal  : Dictionary.Symbol;
      Formal_Kind     : Dictionary.GenericParameterKind;
      Expression_Node : STree.SyntaxNode;
   begin
      -- Check that each named assoication appears and appears once only
      CheckNamedAssociation (The_Formals               => Generic_Sym,
                             Scope                     => Scope,
                             Named_Argument_Assoc_Node => Nam_Arg_Ass_Node);

      -- now process each formal in turn
      Formal_It := Dictionary.FirstGenericFormalParameter (Generic_Sym);
      while not Dictionary.IsNullIterator (Formal_It) loop
         --# assert STree.Table = STree.Table~ and
         --#   Syntax_Node_Type (Nam_Arg_Ass_Node, STree.Table) = SP_Symbols.named_argument_association;
         -- gather up formal data
         Current_Formal := Dictionary.CurrentSymbol (Formal_It);

         -- gather up actual data
         Expression_Node := FindActualNode (For_Formal                => Current_Formal,
                                            Named_Argument_Assoc_Node => Nam_Arg_Ass_Node);
         -- ASSUME Expression_Node = expression OR NULL
         --# check Syntax_Node_Type (Expression_Node, STree.Table) = SP_Symbols.expression or
         --#   Expression_Node = STree.NullNode;
         if Syntax_Node_Type (Node => Expression_Node) = SP_Symbols.expression then
            -- ASSUME Expression_Node = expression
            -- wff them together here
            Formal_Kind := Dictionary.GetGenericFormalParameterKind (Current_Formal);
            case Formal_Kind is
               when Dictionary.GenericTypeParameter =>
                  Process_Generic_Type_Parameter
                    (Formal_Sym        => Current_Formal,
                     Actual_Expression => Expression_Node,
                     Instantiation_Sym => Instantiation_Sym,
                     Scope             => Scope,
                     Error_Found       => Error_Found);
               when Dictionary.GenericObjectParameter =>
                  Process_Generic_Object_Parameter
                    (Formal_Sym        => Current_Formal,
                     Actual_Expression => Expression_Node,
                     Instantiation_Sym => Instantiation_Sym,
                     Scope             => Scope,
                     Error_Found       => Error_Found);
            end case;
         end if;
         Formal_It := Dictionary.NextSymbol (Formal_It);
      end loop;
   end Process_Named_Argument_Association;

   ---------------------------

   procedure Process_Positional_Argument_Association
     (Pos_Arg_Ass_Node  : in     STree.SyntaxNode;
      Generic_Sym       : in     Dictionary.Symbol;
      Instantiation_Sym : in     Dictionary.Symbol;
      Scope             : in     Dictionary.Scopes;
      Error_Found       : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         Error_Found,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Generic_Sym,
   --#                                         GlobalComponentData,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Pos_Arg_Ass_Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Generic_Sym,
   --#                                         GlobalComponentData,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Pos_Arg_Ass_Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Pos_Arg_Ass_Node, STree.Table) = SP_Symbols.positional_argument_association;
   --# post STree.Table = STree.Table~;
   is
      Formal_It       : Dictionary.Iterator;
      Actual_It       : STree.Iterator;
      Current_Formal  : Dictionary.Symbol;
      Formal_Kind     : Dictionary.GenericParameterKind;
      Expression_Node : STree.SyntaxNode;
   begin
      Formal_It       := Dictionary.FirstGenericFormalParameter (Generic_Sym);
      Actual_It       :=
        Find_First_Node (Node_Kind    => SP_Symbols.expression,
                         From_Root    => Pos_Arg_Ass_Node,
                         In_Direction => STree.Down);
      Expression_Node := Pos_Arg_Ass_Node; -- default value in case loop below is skipped

      while not Dictionary.IsNullIterator (Formal_It) and then not STree.IsNull (Actual_It) loop
         -- gather up actual data
         Expression_Node := Get_Node (It => Actual_It);
         --# assert STree.Table = STree.Table~ and
         --#   Syntax_Node_Type (Expression_Node, STree.Table) = SP_Symbols.expression and
         --#   Expression_Node = Get_Node (Actual_It);

         -- gather up formal data
         Current_Formal := Dictionary.CurrentSymbol (Formal_It);
         Formal_Kind    := Dictionary.GetGenericFormalParameterKind (Current_Formal);

         -- wff them together here
         case Formal_Kind is
            when Dictionary.GenericTypeParameter =>
               Process_Generic_Type_Parameter
                 (Formal_Sym        => Current_Formal,
                  Actual_Expression => Expression_Node,
                  Instantiation_Sym => Instantiation_Sym,
                  Scope             => Scope,
                  Error_Found       => Error_Found);
            when Dictionary.GenericObjectParameter =>
               Process_Generic_Object_Parameter
                 (Formal_Sym        => Current_Formal,
                  Actual_Expression => Expression_Node,
                  Instantiation_Sym => Instantiation_Sym,
                  Scope             => Scope,
                  Error_Found       => Error_Found);
         end case;
         Formal_It := Dictionary.NextSymbol (Formal_It);
         Actual_It := STree.NextNode (Actual_It);
      end loop;

      -- completeness check, both loops should run out at same time
      if not Dictionary.IsNullIterator (Formal_It) or else not STree.IsNull (Actual_It) then
         Error_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 635,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Expression_Node),
            Id_Str    => Dictionary.GetSimpleName (Generic_Sym));
      end if;
   end Process_Positional_Argument_Association;

begin -- Wf_Generic_Actual_Part
   Error_Found      := False;
   Actual_Part_Node := Next_Sibling (Current_Node => Generic_Node);
   -- ASSUME Actual_Part_Node = generic_actual_part OR NULL
   SystemErrors.RT_Assert
     (C       => Actual_Part_Node = STree.NullNode
        or else Syntax_Node_Type (Node => Actual_Part_Node) = SP_Symbols.generic_actual_part,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Actual_Part_Node = generic_actual_part OR NULL in Wf_Generic_Actual_Part");
   Check_Missing_Actual_Part
     (Generic_Node     => Generic_Node,
      Actual_Part_Node => Actual_Part_Node,
      Generic_Sym      => Generic_Sym,
      Error_Found      => Error_Found);

   Check_Superfluous_Actual_Part
     (Generic_Node     => Generic_Node,
      Actual_Part_Node => Actual_Part_Node,
      Generic_Sym      => Generic_Sym,
      Error_Found      => Error_Found);

   if not Error_Found and then Syntax_Node_Type (Node => Actual_Part_Node) = SP_Symbols.generic_actual_part then
      -- ASSUME Actual_Part_Node = generic_actual_part
      -- a generic actual part is both present and needed
      Actual_Part_Node := Child_Node (Current_Node => Child_Node (Current_Node => Actual_Part_Node));
      -- ASSUME Actual_Part_Node = named_argument_association OR positional_argument_association
      if Syntax_Node_Type (Node => Actual_Part_Node) = SP_Symbols.named_argument_association then
         -- ASSUME Actual_Part_Node = named_argument_association
         Process_Named_Argument_Association
           (Nam_Arg_Ass_Node  => Actual_Part_Node,
            Generic_Sym       => Generic_Sym,
            Instantiation_Sym => Instantiation_Sym,
            Scope             => Scope,
            Error_Found       => Error_Found);
      elsif Syntax_Node_Type (Node => Actual_Part_Node) = SP_Symbols.positional_argument_association then
         -- ASSUME Actual_Part_Node = positional_argument_association
         Process_Positional_Argument_Association
           (Pos_Arg_Ass_Node  => Actual_Part_Node,
            Generic_Sym       => Generic_Sym,
            Instantiation_Sym => Instantiation_Sym,
            Scope             => Scope,
            Error_Found       => Error_Found);
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Actual_Part_Node = named_argument_association OR " &
              "positional_argument_association in Wf_Generic_Actual_Part");
      end if;
   end if;
end Wf_Generic_Actual_Part;
