-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure CheckNoOverloadingFromTaggedOps
  (Spec_Node     : in STree.SyntaxNode;
   Subprog_Sym   : in Dictionary.Symbol;
   Scope         : in Dictionary.Scopes;
   Abstraction   : in Dictionary.Abstractions;
   Is_Overriding : in Boolean) is
   Root_Subprog_Sym             : Dictionary.Symbol;
   Actual_Tagged_Parameter_Type : Dictionary.Symbol;
   Root_Op_Kind                 : Dictionary.KindsOfOp;

   function SuccessfullyOverrides
     (Root_Subprog, Second_Subprog, Actual_Tagged_Parameter_Type : Dictionary.Symbol)
     return                                                       Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;

   ------------------------------------------------------------------------

   -- given a node which is a subprogram specification, locate and return the lex string
   -- representing the subprogram name
   function Get_Subprogram_Ident (Node : STree.SyntaxNode) return LexTokenManager.Lex_String
   --# global in STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.procedure_specification or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.function_specification or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.entry_specification;
   is
      Result : LexTokenManager.Lex_String := LexTokenManager.Null_String;
   begin
      -- ASSUME Node = procedure_specification OR function_specification OR entry_specification
      if Syntax_Node_Type (Node => Node) = SP_Symbols.procedure_specification
        or else Syntax_Node_Type (Node => Node) = SP_Symbols.function_specification then
         -- ASSUME Node = procedure_specification OR function_specification
         Result := Node_Lex_String (Node => Child_Node (Current_Node => Child_Node (Current_Node => Node)));
      elsif Syntax_Node_Type (Node => Node) = SP_Symbols.entry_specification then
         -- ASSUME Node = entry_specification
         Result :=
           Node_Lex_String
           (Node => Child_Node (Current_Node => Child_Node (Current_Node => Child_Node (Current_Node => Node))));
      end if;
      return Result;
   end Get_Subprogram_Ident;

begin -- CheckNoOverloadingFromTaggedOps

   -- if a potentially inheritable subprogram of the same name exists then
   -- the new declaration is only legal if it successfully overrides it
   Dictionary.SearchForInheritedOperations
     (Name             => Get_Subprogram_Ident (Node => Spec_Node),
      Scope            => Scope,
      Prefix           => Dictionary.NullSymbol,
      Context          => Dictionary.ProofContext,
      OpSym            => Root_Subprog_Sym,
      KindOfOp         => Root_Op_Kind,
      ActualTaggedType => Actual_Tagged_Parameter_Type);
   if Root_Subprog_Sym /= Dictionary.NullSymbol and then Root_Op_Kind /= Dictionary.NotASubprogram then
      -- An inheritable subprogram has been found.
      -- This declaration is only legal if it overrides it
      if not SuccessfullyOverrides
        (Root_Subprog                 => Root_Subprog_Sym,
         Second_Subprog               => Subprog_Sym,
         Actual_Tagged_Parameter_Type => Actual_Tagged_Parameter_Type) then
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 829,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Spec_Node),
            Sym       => Root_Subprog_Sym,
            Scope     => Scope);
         if not Dictionary.IsProofFunction (Subprog_Sym) then
            Dictionary.SetSubprogramSignatureNotWellformed (Abstraction, Subprog_Sym);
         end if;
      elsif CommandLineData.Content.Language_Profile = CommandLineData.SPARK2005
        and then not Is_Overriding
        and then Root_Op_Kind /= Dictionary.NotASubprogram then
         -- An inherited sub-program but its declarations contradicts the
         -- its overriding indicator.
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 844,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Spec_Node),
            Sym       => Subprog_Sym,
            Scope     => Scope);
      end if;
   elsif CommandLineData.Content.Language_Profile = CommandLineData.SPARK2005
     and then Root_Subprog_Sym = Dictionary.NullSymbol
     and then Is_Overriding then
      ErrorHandler.Semantic_Error_Sym
        (Err_Num   => 845,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Spec_Node),
         Sym       => Subprog_Sym,
         Scope     => Scope);
   end if;
end CheckNoOverloadingFromTaggedOps;
