// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


//
// star.cc - The Star class.
//

#include "star.h"
#include "greek.h"

#define NEED_FULL_NAMES
#include "constellations.h"

// For the Star class ------------------------------------------

// Private function

// Function to set a bunch of star data to save typing in the copy constructor
//  and operator= functions.
void Star::Obtain(StringList Names, StringList Membership, StringList Comments,
		  SolidAngle GridPosn, double Distance, double Diameter,
		  double PrimaryDistance, double Magnitude, SpecClass Spectrum,
		  unsigned int Place, unsigned int XPixel, unsigned int YPixel,
		  unsigned int RPixel, bool SLabelDraw)
{
  sNames = Names, sMembership = Membership, sComments = Comments;
  sGridPosn = GridPosn, sDistance = Distance, sDiameter = Diameter;
  sPrimaryDistance = PrimaryDistance, sMagnitude = Magnitude;
  sSpectrum = Spectrum, sPlace = Place, xPixel = XPixel, yPixel = YPixel;
  rPixel = RPixel; sLabelDraw = SLabelDraw;
  return;
}


// Public functions

// Default constructor
Star::Star()
{
  Obtain(StringList(), StringList(), StringList(), SolidAngle(0,0),
	 0.0, 0.0, 0.0, 0.0, SpecClass(), 0, 0, 0, 0, true);
}

// Copy constructor
Star::Star(const Star &s)
{
  Obtain(s.sNames, s.sMembership, s.sComments, s.sGridPosn, s.sDistance,
	 s.sDiameter, s.sPrimaryDistance, s.sMagnitude, s.sSpectrum,
	 s.sPlace, s.xPixel, s.yPixel, s.rPixel, s.sLabelDraw);
}

// Assignment operator
Star & Star::operator = (const Star &s)
{
  if (! (this == &s)) { // no point in copying if objects already the same
    Obtain(s.sNames, s.sMembership, s.sComments, s.sGridPosn, s.sDistance,
	   s.sDiameter, s.sPrimaryDistance, s.sMagnitude, s.sSpectrum,
	   s.sPlace, s.xPixel, s.yPixel, s.rPixel, s.sLabelDraw); 
  }
  return *this;
}


// Constructor to create a Star from a formatted text record.  Text record
//  is assumed to be in the following format.  Line breaks are permitted,
//  but only after the semicolon ending a field.
//
//   StarName1,StarName2,...;RA(h,m,s);Dec(d,m,s);Distance(L-Y);Diameter(L-Y);
//   SpectralClass;Magnitude;PrimaryDistance(L-Y);
//   Membership;Comments
//
//  Example (Proxima Centauri):
//
//   Proxima Cen,Alpha Cen C,Rigil Kent C;14,32,0;-62,49,0;4.24;0;M5e V;
//   15.49;0.252;Alpha Cen triple system;Nearest star, known flare star [etc]
//
//  (The diameter field should be zero in the record if not specifically
//  known; it will then be calculated from the spectral class and magnitude.)
//
//  Note: Other field and subfield separators than ';' and ',' may be assumed
//  by changing the definitions of FIELD_DELIMITER and SUBFIELD_DELIMITER
//  in "star.h".
//
//  The ctor "fastconversion" argument, if set to true, will cause only the
//  most fundamental attributes about the star (position and basic
//  spectral class) to be written.  This is for speed.  The default is false.
//
//  The "nameconvert" argument, if true, causes any constellation abbreviations
//  in the name, e.g. "UMa", to be translated into the full genitive form,
//  e.g. "Ursae Majoris".

Star::Star(const char *record, bool fastconversion, bool nameconvert)
{
  StringList fields, rastring, decstring;
  int fieldslength;
  double RA, Dec;

  // First, tokenize the record into fields.
  fields = StringList(record, FIELD_DELIMITER);
  fieldslength = fields.size();

  // Now, extract data from each field.  First the fields for a fast
  //  conversion (where all we want is to see if the star passes the filters):

  // Right ascension and declination: convert from h,m,s and
  //  +/-d,m,s to decimal format, then put into SolidAngle sGridPosn
  rastring = StringList(fields[1], SUBFIELD_DELIMITER);
  decstring = StringList(fields[2], SUBFIELD_DELIMITER);
  
  RA = RAStringsToRadians(rastring[0], rastring[1], rastring[2]);
  Dec = DecStringsToRadians(decstring[0], decstring[1], decstring[2]);
  sGridPosn = SolidAngle(RA, Dec);

  // Other fields
  sDistance = myatof(fields[3]);
  sSpectrum = SpecClass(fields[5]);

  if (fastconversion) /* then bail out here */
    return;

  // Then everything else:

  sSpectrum.initialize();
  sMagnitude = myatof(fields[6]);

  sNames = StringList(fields[0], SUBFIELD_DELIMITER);
  sNames.stripspace();

  if (nameconvert) {
    // translate constellation abbrevs. to genitive names
    for (unsigned int i = 0; i < sNames.size(); i++) {
      StringList temp = StringList(sNames[i], ' ');
      for (unsigned int j = 1; j < temp.size(); j++)
        if (temp.strlen(j) == 3)
	  for (unsigned int k = 0; k < NUM_CONSTELLATIONS; k++)
	    if (strcmp(temp[j], constellations[k]) == 0) {
	      temp.set(constelnames[k], j);
	      sNames.set(temp.flatten(' '), i);
	      goto next_name;
	    }
      next_name: continue;
    }
  }

  sDiameter = myatof(fields[4]);
  if (sDiameter <= 0.0)
    sDiameter = sSpectrum.diameter(sMagnitude);

  sPrimaryDistance = myatof(fields[7]);
  sMembership = StringList(fields[8], SUBFIELD_DELIMITER);
  sMembership.stripspace();
  sComments = StringList(fields[9]);
  sComments.stripspace();

  sPlace = xPixel = yPixel = rPixel = 0;
}


// function to determine whether the star should be included in a StarArray
//  with the set of rules given in "rules".
//
//  Note: If star is a secondary star "too close" to its primary, it should
//  be be put into the array ONLY if it passes the filter but its primary
//  doesn't.  This filtering occurs in StarArray::Read(), NOT here.

bool Star::PassesFilter(const Rules &rules) const
{
  // Filtering by magnitude is already done by the first filter in
  //  StarArray::Read().

  // Filter by location
  Vector relativeLocation = GetStarXYZ() - rules.ChartLocation;
  if (relativeLocation.magnitude() > rules.ChartRadius)
    return false;

  // Filter by spectral class
  if (! rules.StarClasses[SpecHash(GetStarClass().getCoarseType())])
    return false;

  // if we get down here, star passed all the filters.
  return true;
}


// Display(): plot the star onto the painting device, as viewed from a
//  specified point and orientation.  (We assume that the star coordinates
//  and chart center given are in the same coordinate system.)
//
//  Note: StarViewer is a generic display class defined in viewer.h -- I am
//  trying to keep the graphics-library dependent functions contained in
//  descendant classes of StarViewer (e.g. KDEViewer, GTKViewer, etc.,
//  which will be wrapper classes around the given graphics libraries).
//  This will make the code a bit more confusing, but make the program
//  easier to port.

void Star::Display(const Rules &rules, StarViewer *sv) const
{
  int wincenterX, wincenterY, starBaseY;
  unsigned int windowsize, pixelradius;

  Vector relativeLocation;
  double x, y, z;
  unsigned long barcolor;

  Vector center = rules.ChartLocation;
  double radius = rules.ChartRadius;
  SolidAngle orientation = rules.ChartOrientation;
  bool bar = rules.StarBars;

  // Determine radius and center of chart, in pixels
  windowsize = (sv->width() > sv->height()) ? sv->height() : sv->width();
  pixelradius = (int)(0.4 * windowsize);
  wincenterX = sv->width() / 2;
  wincenterY = sv->height() / 2;

  // Determine star position in "local coordinates", where:
  //  XZ-plane is vertical and perpendicular to the computer screen,
  //   with X-axis pointing out of the screen and tipped DOWNward
  //   by the angle orientation.getTheta()
  //  Y-axis is horizontal, parallel to the screen, and pointing rightward
  //  Distances are in pixels

  relativeLocation = GetStarXYZ() - center;
  x = relativeLocation.getX() * cos(orientation.getPhi())
    + relativeLocation.getY() * sin(orientation.getPhi());
  y = -relativeLocation.getX() * sin(orientation.getPhi())
    + relativeLocation.getY() * cos(orientation.getPhi());
  z = relativeLocation.getZ();
  relativeLocation = Vector(x,y,z) * pixelradius / radius;

  // Determine 2-D projection of this relative location onto the screen.
  xPixel = wincenterX + (int)relativeLocation.getY();
  starBaseY = wincenterY
    + (int)(relativeLocation.getX() * sin(orientation.getTheta()));
  yPixel = starBaseY
    - (int)(relativeLocation.getZ() * cos(orientation.getTheta()));
  
  // Determine how large the star should be drawn.
  if (sDiameter * pixelradius / (2 * radius) > STAR_PIXEL_RADIUS) {
    rPixel = (int)(sDiameter * pixelradius / (2 * radius));
    if (rPixel > pixelradius) rPixel = pixelradius;
  }
  else
    rPixel = 0;

  // Draw the star and accompanying devices (position bar, label).

  barcolor = (relativeLocation.getZ() >= 0.0) ? POSITIVE : NEGATIVE;

  if (bar) {
    if (relativeLocation.getZ() * orientation.getTheta() >= 0.0) {
      // then star is in the hemisphere of the chart facing us, so:
      // draw the reference ellipse on the x-y plane
      sv->setcolor(barcolor);
      sv->setfill(false);
      sv->drawellipse(xPixel, starBaseY, 3, 2);
      // draw the vertical bar
      sv->setcolor(BACKGROUND);
      sv->drawline(xPixel - 1, starBaseY, xPixel - 1, yPixel);
      sv->drawline(xPixel + 1, starBaseY, xPixel + 1, yPixel);
      sv->setcolor(barcolor);
      sv->drawline(xPixel, starBaseY, xPixel, yPixel);
    }

    else {
      // outline the vertical bar
      sv->setcolor(BACKGROUND);
      sv->drawline(xPixel - 1, starBaseY, xPixel - 1, yPixel);
      sv->drawline(xPixel + 1, starBaseY, xPixel + 1, yPixel);
    }
  }

  // draw the star and label
  rPixel = Draw(rules, sv, xPixel, yPixel, rPixel);
  // make sure the effective mouse-click radius isn't too small:
  if (rPixel < STAR_PIXEL_RADIUS) rPixel = STAR_PIXEL_RADIUS;

  if (bar && relativeLocation.getZ() * orientation.getTheta() < 0.0) {
    // draw the vertical bar
    sv->setcolor(barcolor);
    sv->drawline(xPixel, starBaseY, xPixel, yPixel);
    // draw the ellipse
    sv->setcolor(barcolor);
    sv->setfill(false);
    sv->drawellipse(xPixel, starBaseY, 3, 2);
  }

  return;
}


// Draw(): plot the star onto the painting device at a given pixel location,
//  not worrying about coordinates, etc.  Returns the radius of circle drawn,
//  in pixels.

unsigned int Star::Draw(const Rules &rules, StarViewer *sv, 
			int xposn, int yposn, int pixelradius) const
{
  // set size of star to pixelradius, unless pixelradius is <= 0 / not given:

  if (pixelradius <= 0) {
    pixelradius = STAR_PIXEL_RADIUS;

    if (rules.StarDiameters == MK_DIAMETERS) {
      // Increase size of the circle for giant-class stars.
      double mktype = sSpectrum.getMKtype();
      if (mktype != 0.0) {
	if (mktype <= 3.5)
	  pixelradius++;
	if (mktype <= 1.5)
	  pixelradius++;
      }
    }

    else if (rules.StarDiameters == MAGNITUDE_DIAMETERS) {
      // Set size of the circle based upon the absolute magnitude.
      if (sMagnitude >= 8.0) pixelradius = 1;
      else if (sMagnitude < -4.0) pixelradius = 5;
      else pixelradius = (int)(4.0 - sMagnitude / 4.0);
    }
  }

  // Draw the circle
  sv->setfill(true);
  sv->setcolor(sSpectrum.color());
  sv->drawstar(xposn, yposn, pixelradius);  

  if (rules.StarLabels != NO_LABEL) { // number the star appropriately
    char starPlace[5];
    Labeltype label = rules.StarLabels;

    // make non-stellar object labels a different color
    sv->setcolor((sSpectrum.getCoarseType() == '*') ? NON_STELLAR_COLOR : 
		 TEXT_COLOR);

    sv->setfill(false);

    switch (label) {
    case NUMBER_LABEL:
      snprintf(starPlace, 5, "%d", sPlace);
      sv->drawtext(starPlace,
		   xposn + (int)(pixelradius / M_SQRT2) + 3,
		   yposn - (int)(pixelradius / M_SQRT2) - 3, 10);
      break;
    case STRING_LABEL:
      sv->drawtext(sNames[0],
		   xposn + (int)(pixelradius / M_SQRT2) + 3,
		   yposn - (int)(pixelradius / M_SQRT2) - 3, 10);
      break;
    case LANDMARK_LABEL:
      if (sLabelDraw || (strcmp(sNames[0], "Sun") == 0))
	sv->drawtext(sNames[0],
		     xposn + (int)(pixelradius / M_SQRT2) + 3,
		     yposn - (int)(pixelradius / M_SQRT2) - 3, 10);
      break;
    default: // do nothing
      break;
    }
  }

  return pixelradius;
}


// function to take star information, format it properly, and append it
//  as strings to a StringList.

StringList Star::GetInfo(const Rules &rules, bool punctuation, char sep) const
{
  StringList output = StringList();
  char RA[16], Dec[16], temp[13];

  snprintf(temp, 10, "%d", sPlace);       // 0. Label number
  output.append(temp);

  // 1. Most common designation
  output.append((sNames.size()) ? sNames[0] : "");
  
  if (sDistance == 0.0) {
    output.append("N/A");
    output.append("N/A");
  }
  else {
    // 2. { Right ascension | Galactic longitude }
    RadiansToRAString(sGridPosn.getPhi(), RA, rules.CelestialCoords,
		      punctuation, sep);
    output.append(RA);

    // 3. { Declination | Galactic latitude }
    RadiansToDecString(sGridPosn.getTheta(), Dec, punctuation, sep);
    output.append(Dec);
  }

  snprintf(temp, 12, "%.6g", sDistance); // 4. Distance in L-Y
  output.append(temp);

  output.append(sSpectrum.print());      // 5. Spectral class

  snprintf(temp, 5, "%.1f", sMagnitude); // 6. Absolute visual magnitude
  output.append(temp);

  output.append(sMembership[0]);         // 7. Cluster membership
  output.append(sComments[0]);           // 8. Comments

  snprintf(temp, 12, "%.2g", sPrimaryDistance); // 9. Distance from primary
  output.append(temp); 
  
  for (unsigned int i = 1; i < sNames.size(); i++)
    output.append(sNames[i]);                   // 10+. Other names for star

  return output;
}
