/*
 * stklos.c	-- STklos interpreter main function
 * 
 * Copyright  1999-2000 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 * 
 *           Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: 28-Dec-1999 21:19 (eg)
 * Last file update: 13-Sep-2000 11:38 (eg)
 */

#include <stklos.h>
#ifdef HAVE_XGETOPT_LONG
#  include <getopt.h>
#else
#  include "../compat/getopt.h"
#endif


#define ADD_OPTION(o, k)  					\
  if (*o) options = STk_key_set(options, 			\
				STk_makekey(k), 		\
				STk_Cstring2string(o));


/*=============================================================================
 *
 * Program arguments
 *
 *=============================================================================
 */

static char *boot_file    = DEFAULT_BOOT_FILE;
static char *program_file = "";
static char *load_file    = "";
static char *sexpr        = "";


static struct option long_options [] = 
{
  {"version",   no_argument, 	   NULL, 'v'},
  {"file",      required_argument, NULL, 'f'},
  {"load",      required_argument, NULL, 'l'},
  {"execute",   required_argument, NULL, 'e'},
  {"boot-file", required_argument, NULL, 'b'},
  {"help",      no_argument, 	   NULL, 'h'},
  {NULL}
};

void Usage(char *progname, int only_version)
{
  fprintf(stderr, "%s (version %s)\n", progname, VERSION);
  if (only_version) return;
  fprintf(stderr, "Usage: %s [options] progname [arg ... ]", progname);
  fprintf(stderr, "
Possible options:
	-l file, --load=file		load 'file' before going interactive
    	-f file, --file=file		use 'file' as program
	-e sexpr, --execute=sexpr	evaluate the given sexpr and exit
	-b file, --boot-file=file	use 'file' to boot the system
	-v, --version			print program version and exit
	-h, --help			print this help and exit\n");
}


int process_program_arguments(int argc, char *argv[])
{
  extern char *optarg;
  extern int optind;
  int c;

  for ( ; ; ) {
    c = getopt_long(argc, argv, "vhf:l:e:b:", long_options, NULL);
    if (c == -1) break;
    
    switch (c) {
      case 'v': Usage(*argv, 1); exit(0);
      case 'f': program_file = optarg; break;
      case 'l': load_file    = optarg; break;
      case 'e': sexpr	     = optarg; break;
      case 'b': boot_file    = optarg; break;
      case '?': /* message error is printed by getopt */
		fprintf(stderr, "Try `%s --help' for more infomation\n", *argv);
		exit(1);
      default:  Usage(*argv, 0); exit(c != 'h');
    }
  }
  return optind;
}

static void  build_scheme_args(int argc, char *argv[])
{
  SCM options, l = STk_nil;
  int i;

  for (i = argc-1; i >= 0; i--)
    l = STk_cons(STk_Cstring2string(argv[i]), l);

  options = LIST2(STk_makekey(":argv"), l);
  ADD_OPTION(program_file, ":file");
  ADD_OPTION(load_file,    ":load");
  ADD_OPTION(sexpr, 	   ":sexpr");

  STk_define_variable(STk_intern("*%program-args*"), options, STk_current_module);
}

int main(int argc, char *argv[])
{
  int ret;

  /* Process command arguments */
  ret = process_program_arguments(argc, argv);
  argc -= ret;
  argv += ret;
  
  /* Initialize the library */
  if (!STk_init_library(&argc, &argv)) {
    fprintf(stderr, "cannot initialize the STklos library\nABORT\n");
    exit(1);
  }

  /* Place the interpreter arguments in the Scheme variable *%program-args* */
  build_scheme_args(argc, argv);

  /* Boot the VM */
  ret = STk_load_boot(boot_file);
  if (ret < 0) {
    fprintf(stderr, "cannot boot with \"%s\" file (code=%d)\n", boot_file, ret);
    exit(1);
  }

  /* Bye */
  printf("; Leaving STklos\n");
  return ret;
}
