/*
 * f p o r t . c				-- File ports
 *
 * Copyright  2000-2003 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 *           Author: Erick Gallesio [eg@unice.fr]
 *    Creation date:  8-Jan-2000 14:48 (eg)
 * Last file update: 10-Feb-2003 11:15 (eg)
 *
 * This implementation is built by reverse engineering on an old SUNOS 4.1.1
 * stdio.h. It has been simplified to fit the needs for STklos. In particular
 * non buffered file are not implemented. Anyway this is faster than an
 * implementation using the C buffered IO (at least on glibc)
 *
 */
#include "stklos.h"


char *STk_current_filename;		  /* Name of the file we read */
int STk_line_counter, STk_column_counter; /* Position when reading a file */

SCM STk_curr_iport, STk_curr_oport, STk_curr_eport;  /* current active ports   */
SCM STk_stdin, STk_stdout, STk_stderr;		     /* The unredirected ports */


/*
 * Since we manage ourselves our port, we need to kee a reference on all
 * the open ports such that all the port are flushed when we exit from
 * the program.
 * FIXME: This method is not optimal. What we should have is a finalizer on
 * ports (that is what we have til 0.51 version) AND a weak list of the open
 * ports. With a weak list port can be closed on finalization, whereas in our
 * solution, unclosed port which can be garbaged will not be freed because of 
 * the reference through the global all_fports
 */
static SCM all_fports = STk_nil;

/*===========================================================================*\
 * 
 * Low level plugins
 * 
\*===========================================================================*/


#define TTY_BUFSIZE   	128
#define OTHER_BUFSIZE 	4096


#define STK_IOFBF	(1 << 0) /* Full buffered*/
#define STK_IOLBF	(1 << 1) /* Line buffered */
#define STK_IONBF	(1 << 2) /* No buffered (unused for now) */
#define STK_IOEOF	(1 << 3) /* EOF encountered on this file */


struct fstream {
  unsigned char *base;  /* buffer start */
  unsigned char *ptr;   /* ptr on the current character in buffer  */
  int	cnt;	        /* # of chars in the buffer */
  int	bufsize;        /* buffer size */
  int   stream_flags;   /* flags */ 
  FILE  *f;             /* the file itself */
  short	fd;	        /* file descriptor */
  char *filename;
  SCM  read_event;
  SCM  write_event;
  SCM  idle_procs;
};


#define PORT_BASE(x)    	(((struct fstream *) (x))->base)
#define PORT_PTR(x)     	(((struct fstream *) (x))->ptr)
#define PORT_CNT(x)     	(((struct fstream *) (x))->cnt)
#define PORT_BUFSIZE(x) 	(((struct fstream *) (x))->bufsize)
#define PORT_STREAM_FLAGS(x) 	(((struct fstream *) (x))->stream_flags)
#define PORT_FILE(x)  		(((struct fstream *) (x))->f)
#define PORT_FD(x)  		(((struct fstream *) (x))->fd)
#define PORT_NAME(x)		(((struct fstream *) (x))->filename)
#define PORT_REVENT(x)		(((struct fstream *) (x))->read_event)
#define PORT_WEVENT(x)		(((struct fstream *) (x))->write_event)
#define PORT_IDLE(x)		(((struct fstream *) (x))->idle_procs)


/*===========================================================================*\
 * 
 * 					Utils
 * 
\*===========================================================================*/

static void error_bad_file_name(SCM name)
{
  STk_error("bad file name ~S", name);
}

static void error_bad_file_port(SCM v)
{
  STk_error("bad file port ~S", v);
}

/*=============================================================================*/

static void fill_buffer(struct fstream *f)
{
  int n = 0; 				/* to avoid gcc warning */
  unsigned char *ptr = PORT_BASE(f);
  
  do
    n = read(PORT_FD(f), ptr, PORT_BUFSIZE(f));
  while ((n == -1) && (errno == EINTR));
  
  if (n == 0) {
    PORT_STREAM_FLAGS(f) |= STK_IOEOF;
  } else {
    PORT_CNT(f) = n;
    PORT_PTR(f) = ptr;
  }
}

/*=============================================================================*/

static int flush_buffer(struct fstream *f)
{
  int n, ret;

  n = PORT_CNT(f);
  /* Write buffer */
  ret = write(PORT_FD(f), PORT_BASE(f), n);

  /* Update structure */
  PORT_CNT(f) = 0;
  PORT_PTR(f) = PORT_BASE(f);

  return ret < 0;
}

/*=============================================================================*/

static void empty_buffer(struct fstream *f)
{
  PORT_CNT(f) = 0;
  PORT_PTR(f) = PORT_BASE(f);
}



/*=============================================================================*/

static inline int Feof(void *stream)
{
  return PORT_STREAM_FLAGS(stream) & STK_IOEOF;
}


static inline int Freadyp(void *stream)
{
  if (PORT_CNT(stream) > 0)
    return TRUE;
  else {
    fd_set readfds;
    struct timeval timeout;
    int f = PORT_FD(stream);

    FD_ZERO(&readfds);
    FD_SET(f, &readfds);
    timeout.tv_sec = 0; timeout.tv_usec = 10; /* FIXME: 0=>charge trop importante*/
    return (select(f+1, &readfds, NULL, NULL, &timeout));
  }
}

static inline int Fgetc(void *stream)
{
  if (PORT_IDLE(stream) != STk_nil) {
    /* There is at least an idle handler */
    SCM idle;
    
    while (!Freadyp(stream))
      for (idle = PORT_IDLE(stream); !NULLP(idle); idle = CDR(idle))
	STk_C_apply(CAR(idle), 0);
  }

  for ( ; ; ) {
    if (--PORT_CNT(stream) >= 0)
      return (int) *PORT_PTR(stream)++;
    else {
      fill_buffer(stream);
      if (Feof(stream)) return EOF;
    }
  }
}


static inline int Fread(void *stream, void *buff, int count)
{
  return read(PORT_FD(stream), buff, count);
}

static int Fclose(void *stream)		/* File version */
{
  int ret = flush_buffer(stream);

  return (ret == EOF) ? EOF : close(PORT_FD(stream));
}

static int Fclose_pipe(void *stream)	/* pipe version (used by "| cmd" files */
{
  int ret = flush_buffer(stream);

  return (ret == EOF) ? EOF : pclose(PORT_FILE(stream));
}


static inline int Fputc(int c, void *stream)
{
  int ret = c;

  for ( ; ; ) {
    if (PORT_CNT(stream) < PORT_BUFSIZE(stream)) {
      *PORT_PTR(stream)++ = (unsigned char) c;
      PORT_CNT(stream)   += 1;
      if (c == '\n' && PORT_STREAM_FLAGS(stream) & STK_IOLBF) {
	if (flush_buffer(stream) != 0)
	  return EOF;
      }
      break;
    }
    else {
      /* buffer is full */
      if (flush_buffer(stream) != 0)
	return EOF;
    }
  }
  return ret;
}


static int Fwrite(void *stream, void *buff, int count)
{
  return write(PORT_FD(stream), buff, count);
}


static int Fputs(char *s, void *stream)
{
  while (*s) {
    if (Fputc((unsigned char) *s++, stream) == EOF) 
      return EOF;
  }
  return 0;
}


static int inline Fflush(void *stream)
{
  return flush_buffer(stream);		/* Problem if file opened for reading */
}


static void fport_print(SCM obj, SCM port) 	/* Generic printing of file ports */
{
  char buffer[MAX_PATH_LENGTH + 20];

  sprintf(buffer, "#[%s-port '%s' (%d)%s]", 
	  IPORTP(obj) ? "input" : "output",
	  PORT_NAME(PORT_STREAM(obj)), PORT_FD(PORT_STREAM(obj)),
	  PORT_IS_CLOSEDP(obj) ? " (closed)" : "");
  STk_puts(buffer, port);
}


static void fport_release(SCM port)	/* remove port from list of open ports */
{
  all_fports = STk_dremq(port, all_fports);
}



#ifdef STK_PORT_FINALIZER
static void fport_finalizer(struct port_obj *port)
{
  /* Close the associated stream */
  STk_close((SCM) port);
}
#endif

static struct port_obj *
make_fport(char *filename, FILE *f, int flags)
{
  struct fstream  *fs = STk_must_malloc(sizeof(struct fstream));
  int n, mode, fd;
  SCM res;

  fd = fileno(f);
  /* allocate buffer for file */
  if (isatty(fd)) {
    n      = TTY_BUFSIZE;
    mode   = STK_IOLBF;
    flags |= PORT_IS_INTERACTIVE;
  } else {
    n    = OTHER_BUFSIZE;
    mode = STK_IOFBF;
  }

  /* Initialize the stream part */
  PORT_BASE(fs)    	 = STk_must_malloc_atomic(n);
  PORT_PTR(fs)           = PORT_BASE(fs);
  PORT_CNT(fs)		 = 0;
  PORT_BUFSIZE(fs) 	 = n;
  PORT_STREAM_FLAGS(fs)  = mode;
  PORT_FILE(fs)		 = f;
  PORT_FD(fs)		 = fd;
  PORT_NAME(fs)		 = STk_strdup(filename);
  PORT_REVENT(fs)	 = STk_false;
  PORT_WEVENT(fs)	 = STk_false;
  PORT_IDLE(fs)		 = STk_nil;

  /* Initialize now the port itsef */
  NEWCELL(res, port);

  PORT_STREAM(res)	= fs;
  PORT_FLAGS(res)	= flags;
  PORT_UNGETC(res) 	= EOF;
  PORT_LINE(res)	= 1;
  PORT_POS(res)		= 0;

  PORT_PRINT(res)	= fport_print;
  PORT_RELEASE(res)	= fport_release;
  PORT_GETC(res)	= Fgetc;
  PORT_READY(res)	= Freadyp;
  PORT_EOFP(res)	= Feof;
  PORT_CLOSE(res)	= (flags & PORT_IS_PIPE) ? Fclose_pipe: Fclose;
  PORT_PUTC(res)	= Fputc;
  PORT_PUTS(res)	= Fputs;
  PORT_FLUSH(res)	= Fflush;
  PORT_BREAD(res)	= Fread;
  PORT_BWRITE(res)	= Fwrite;

#ifdef STK_PORT_FINALIZER
  /* Add a finalizer on file to close it when the GC frees it */
  STk_register_finalizer(res, fport_finalizer);
#endif
  /* Add res to the list of open ports */ 
  all_fports = STk_cons(res, all_fports);
  
  return (struct port_obj *)res;
}


#ifdef WIN32
static char *convert_for_win32(char *mode)
{
  /* Things are complicated on Win32 (as always). So we onvert all files 
   * in binaries files. Note that this function is not called when we work 
   * on ports since they only accept version without "b" on Cygwin
   */
  switch (*mode) {
    case 'r': if (mode[1] == '\0') return "rb";
      	      if (mode[1] == '+')  return "rb+";
	      break;
    case 'w': if (mode[1] == '\0') return "wb";
      	      if (mode[1] == '+')  return "wb+";
	      break;
    case 'a': if (mode[1] == '\0') return "ab";
      	      if (mode[1] == '+')  return "ab+";
	      break;
  }
  return mode;
}
#endif


static SCM open_file_port(SCM filename, char *mode, int flags, int error)
{
  FILE *f;
  char *full_name, *name;

  /* We use fopen (or popen) here to simplify problems with mode opening 
   * But since we don't use the buffer, we say that we work in non buffered 
   * mode
   */
  name = STRING_CHARS(filename);

  if (strncmp(name, "| ", 2)) {
    full_name  = STk_expand_file_name(name);
    flags     |= PORT_IS_FILE;
#ifdef WIN32
    mode = convert_for_win32(mode);
#endif
    if ((f = fopen(full_name, mode)) == NULL) {
      if (error) 
	STk_error("could not open file ~S", filename);
      else
	return STk_false;
    }
  }
  else {
    full_name  = name;
    flags     |= PORT_IS_PIPE;
    if ((f = popen(name+1, mode)) == NULL) {
      if (error) 
	STk_error("could not create pipe for ~S", STk_Cstring2string(name+2));
      else 
	return STk_false;
    }
  }

  /* Don't use (and allocate) a buffer for this file */
  setvbuf(f, NULL, _IONBF, 0);
  return (SCM) make_fport(full_name, f, flags);
}


SCM STk_fd2scheme_port(int fd, char *mode, char *identification)
{
  FILE *f;
  int flags;

  f = fdopen(fd, mode);
  if (!f) return (SCM) NULL;

  /* Don't use (and allocate) a buffer for this file */
  setvbuf(f, NULL, _IONBF, 0);
  
  flags = PORT_IS_FILE | ((*mode == 'r') ? PORT_READ : PORT_WRITE);
  return (SCM) make_fport(identification, f, flags);
}


/*=============================================================================*\
 *                               Open/Close
\*=============================================================================*/

/*
<doc open-input-file
 * (open-input-file filename)
 *
 * Takes a string naming an existing file and returns an input port capable
 * of delivering characters from the file. If the file cannot be opened, 
 * an error is signalled.
 * 
 * @strong{Note}: if |filename| starts with the string @emph{"@pipe "}, 
 * this procedure returns a pipe port. Consequently, it is not possible to
 * open a file whose name starts with those two characters.
doc>
 */
DEFINE_PRIMITIVE("open-input-file", open_input_file, subr1, (SCM filename))
{
  ENTER_PRIMITIVE(open_input_file);

  if (!STRINGP(filename)) error_bad_file_name(filename);
  return open_file_port(filename, "r", PORT_READ, TRUE);
}

/*
<doc open-output-file
 * (open-output-file filename)
 *
 * Takes a string naming an output file to be created and returns an output
 * port capable of writing characters to a new file by that name. If the file 
 * cannot be opened, an error is signalled. If a file with the given name 
 * already exists, it is rewritten.
 *
 * @strong{Note}: if |filename| starts with the string @emph{"@pipe "}, 
 * this procedure returns a pipe port. Consequently, it is not possible to
 * open a file whose name starts with those two characters.
doc>
 */
DEFINE_PRIMITIVE("open-output-file", open_output_file, subr1, (SCM filename))
{
  ENTER_PRIMITIVE(open_output_file);

  if (!STRINGP(filename)) error_bad_file_name(filename);
  return open_file_port(filename, "w", PORT_WRITE, TRUE);
}


/*
<doc EXT input-file-port? output-file-port?
 * (input-file-port? obj)
 * (output-file-port? obj)
 *
 * Returns |#t| if |obj| is a file input port or a file output port respectively, 
 * otherwise returns #f.
doc>
 */
DEFINE_PRIMITIVE("input-file-port?", input_fportp, subr1, (SCM port))
{
  return MAKE_BOOLEAN(IFPORTP(port));
}


DEFINE_PRIMITIVE("output-file-port?", output_fportp, subr1, (SCM port))
{
  return MAKE_BOOLEAN(OFPORTP(port));
}

/*
<doc EXT open-file
 * (open-file filename mode)
 *
 * Opens the file whose name is |filename| with the specified string
 * |mode| which can be: 
 * @itemize @bullet{}
 * @item |"r"| to open file for reading. The stream is positioned at
 * the beginning of the file.
 *
 * @item |"r+"| to open file for reading and writing.  The stream is
 * positioned at the beginning of the file.
 * 
 * @item |"w"| to truncate file to zero length or create file for writing.
 * The stream is positioned at the beginning of the file.
 *
 * @item |"w+"| to open  file for reading and writing. The file is created
 * if it does not exist, otherwise it is truncated. The stream is positioned
 * at the beginning of the file.
 *
 * @item |"a"| to open for writing.  The file is created if  it  does
 * not exist. The stream is positioned at the end of the file.
 *
 * @item |"a+"| to open file for reading and writing. The file is created
 * if it does not exist. The stream is positioned at the end of the file.
 *
 * @end itemize
 * If the file can be opened, |open-file| returns the port associated with
 * the given file, otherwise it returns |#f|. Here again, the ``magic'' 
 * string "@pipe " permits to open a pipe port (in this case mode can only be 
 * |"r"| or |"w"|).
doc>
 */

DEFINE_PRIMITIVE("open-file", scheme_open_file, subr2, (SCM filename, SCM mode))
{
  int type;
  
  ENTER_PRIMITIVE(scheme_open_file);
  if (!STRINGP(filename)) error_bad_file_name(filename);
  if (!STRINGP(mode))     goto Error;
  
  switch (STRING_CHARS(mode)[0]) {
    case 'r': type = (STRING_CHARS(mode)[1] == '+') ? PORT_RW : PORT_READ;  break;
    case 'a':
    case 'w': type = (STRING_CHARS(mode)[1] == '+') ? PORT_RW : PORT_WRITE; break;
    default:  goto Error;
  }
  return open_file_port(filename, STRING_CHARS(mode), type, FALSE);
Error:
  STk_error("bad opening mode ~S", mode);
  return STk_void; /* for the compiler */
}


/*
<doc EXT rewind-file-port
 * (rewind-file-port port)
 *
 * Sets the port position to the beginning of |port|. The value returned by 
 * |rewind-port| is @emph{void}.
doc>
 */
SCM STk_rewind_file_port(SCM port)
{
  struct fstream *stream;

  if (!FPORTP(port)) error_bad_file_port(port);
  
  stream = PORT_STREAM(port);

  if (PORT_FLAGS(port) & (PORT_WRITE | PORT_RW)) flush_buffer(stream);
  else if (PORT_FLAGS(port) & PORT_READ)         empty_buffer(stream);
  
  lseek(PORT_FD(stream), 0, SEEK_SET);
  return STk_void;
}

DEFINE_PRIMITIVE("rewind-file-port", scheme_rewind_file_port, subr1, (SCM port))
{
  ENTER_PRIMITIVE(scheme_rewind_file_port);
  return STk_rewind_file_port(port);
}


/*
<doc EXT port-file-name
 * (port-file-name port)
 *
 * Returns the file name used to open |port|; |port| must be a file port. 
doc>
 */
DEFINE_PRIMITIVE("port-file-name", port_file_name, subr1, (SCM port))
{
  ENTER_PRIMITIVE(port_file_name);
  if (!FPORTP(port)) error_bad_file_port(port);
  return STk_Cstring2string(PORT_NAME(PORT_STREAM(port)));
}


DEFINE_PRIMITIVE("%port-file-fd", port_file_fd, subr1, (SCM port))
{
  ENTER_PRIMITIVE(port_file_fd);
  return FPORTP(port)? MAKE_INT(PORT_FD(PORT_STREAM(port))) : STk_false;
}


DEFINE_PRIMITIVE("%port-idle", port_idle, subr12, (SCM port, SCM val))
{
  ENTER_PRIMITIVE(port_idle);
  if (!FPORTP(port)) error_bad_file_port(port);

  if (val) {
    /* Set the idle list to the given value. No control on the content of 
     * the procedure list (must be done in Scheme)
     */
    if (STk_int_length(val) < 0) STk_error("bad list ~S", val);
    PORT_IDLE(PORT_STREAM(port)) = val;
  }
  return  PORT_IDLE(PORT_STREAM(port));
}


/*===========================================================================*\
 * 
 * Utilities on file ports
 *
\*===========================================================================*/

SCM STk_open_file(char *filename, char *mode)
{
  int type;

  switch (mode[0]) {
    case 'r': type = (mode[1] == '+') ? PORT_RW : PORT_READ;  break;
    case 'a':
    case 'w': type = (mode[1] == '+') ? PORT_RW : PORT_WRITE; break;
    default:  goto Error;
  }
  return open_file_port(STk_Cstring2string(filename), mode, type, FALSE);
Error: 
  STk_panic("bad opening mode %s", mode);
  return STk_void; /* for the compiler */
}

SCM STk_add_port_idle(SCM port, SCM idle_func)
{
  PORT_IDLE(PORT_STREAM(port)) = STk_cons(idle_func, 
					  PORT_IDLE(PORT_STREAM(port)));
  return STk_void;
}

void STk_set_line_buffered_mode(SCM port)
{
  struct fstream* fs;
  int flags;

  /* Assert: port is a valid input port */
  fs    = PORT_STREAM(port);
  flags = PORT_STREAM_FLAGS(fs);

  flags &= ~(STK_IOFBF|STK_IONBF);
  PORT_STREAM_FLAGS(fs) = flags | STK_IOLBF;
}


/*
 * Closing and flushing all open port before exiting
 */
static void at_exit_close_all_fports(void)
{
  SCM l = STk_copy_tree(all_fports); /* Use l since all_fports will be altered */

  while (!NULLP(l)) {
    /* Don't close the input, since it interact badly with interactive REPL */
    if (CAR(l) != STk_curr_iport && CAR(l) != STk_stdin)
      STk_close(CAR(l));
    l = CDR(l);
  }
}


int STk_init_fport(void)
{
  STk_stdin  = STk_curr_iport = (SCM) make_fport("*stdin*",  stdin,
						 PORT_IS_FILE | PORT_READ); 
  STk_stdout = STk_curr_oport = (SCM) make_fport("*stdout*", stdout,
						 PORT_IS_FILE | PORT_WRITE);
  STk_stderr = STk_curr_eport = (SCM) make_fport("*stderr*", stderr,
						 PORT_IS_FILE | PORT_WRITE);

  ADD_PRIMITIVE(open_input_file);
  ADD_PRIMITIVE(open_output_file);

  ADD_PRIMITIVE(input_fportp);
  ADD_PRIMITIVE(output_fportp);
  ADD_PRIMITIVE(scheme_open_file);
  ADD_PRIMITIVE(scheme_rewind_file_port);
  ADD_PRIMITIVE(port_file_name);

  ADD_PRIMITIVE(port_file_fd);
  ADD_PRIMITIVE(port_idle);
  STk_current_filename	= "";	/* "" <=> stdin */


  atexit(at_exit_close_all_fports);
  return TRUE;
}
/*  LocalWords:  filename
 */
