/* This file is part of Strigi Desktop Search
 *
 * Copyright (C) 2006 Jos van den Oever <jos@vandenoever.info>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include <kdebug.h>
#include <kinstance.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kurl.h>
#include <kmimetype.h>
#include <qfileinfo.h>
#include <qtextstream.h>
#include <qstylesheet.h>
#include <sstream>
#include <qimage.h>
#include <kapplication.h>
#include <kcmdlineargs.h>
#include <qeventloop.h>
#include <kio/netaccess.h>
#include <kio/previewjob.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <qbuffer.h>
#include <qregexp.h>
#include "strigi.h"

using namespace KIO;
using namespace std;

class kio_strigiProtocol::Helper : public HtmlHelper {
private:
    QDateTime time;
public:
    const KIconLoader strigiiconloader;

    Helper() {}
    string mapLinkUrl(const string& url, int depth);
    string mapMimetypeIcon(const string& uri,
        const string& mimetype);
    string encodeString(const string& url);
    string escapeString(const string& url);
    string formatDate(time_t date);
    string getCssUrl();
    string highlight(const string& text, const vector<string>& queryterm);
    string mimetypeDescription(const string& mimetype) const;
    string getPathCharacterSeparator() {
        return "<img class='softbreak' src=''/>";
    }
};

string
kio_strigiProtocol::Helper::mapLinkUrl(const string& url, int depth) {
    QFileInfo info(url.c_str());
    if (info.exists()) {
        return "file:"+url;
    } else if (depth == 1) {
        QString u(url);
        u = u.lower();
        if (u.contains(".zip/") || u.contains(".jar/") || u.contains(".war/")) {
            return "zip:"+url;
        }
        if (u.contains(".tar/") || u.contains(".tgz/") || u.contains(".tar.gz/")
                || u.contains(".tar.bz2/")) {
            return "tar:"+url;
        }
    }
    return "jstream:"+url;
}
string
kio_strigiProtocol::Helper::mapMimetypeIcon(const string& uri,
        const string& mimetype) {
    if (mimetype.find('/') == string::npos) {
        return "strigi:icon/application/unknown/128"+uri;
    }
    return "strigi:icon/"+mimetype+"/128"+uri;
/*    QFileInfo info(uri.c_str());
    if (info.exists()) {
        return "thumbnail:"+uri;
    } else {
        QString name = KMimeType::mimeType(mimetype.c_str())->icon(QString::null,0);
        name = "file:"+strigiiconloader.iconPath(name, KIcon::Desktop);
        return (const char*)name.utf8();
    }*/
}
string
kio_strigiProtocol::Helper::encodeString(const string& line) {
    return (const char*)KURL::encode_string(line.c_str()).utf8();
}
string
kio_strigiProtocol::Helper::escapeString(const string& line) {
    return (const char*)QStyleSheet::escape(line.c_str()).utf8();
}
string
kio_strigiProtocol::Helper::formatDate(time_t date) {
    time.setTime_t(date);
    return (const char*)time.toString().utf8();
}
string
kio_strigiProtocol::Helper::getCssUrl() {
    KStandardDirs *dirs = KGlobal::dirs();
    QString file = dirs->findResource("data", "kio_strigi/kio_strigi.css");
    return (const char*)("file://"+file.utf8());
}
string
kio_strigiProtocol::Helper::highlight(const string& text,
        const vector<string>& terms) {
    int pre = 5, post = 5;
    uint maxlen = 200;
    QString t = QString::fromUtf8(text.c_str(), text.length());
    vector<QRegExp> res;
    QString out;
    for (uint i=0; i<terms.size(); ++i) {
        QString term = QString::fromUtf8(terms[i].c_str(), terms[i].length());
        term.replace('*', "\\w*").replace('?',"\\w");
        res.push_back(QRegExp("\\b("+term+")\\b", false));
    }
    int pos = 0;
    int lasts = -1;
    int laste = -1;
    while (pos >= 0 && out.length()+laste-lasts < maxlen) {
        int rep = -1;
        int len = 0;
        for (uint i=0; i<res.size(); ++i) {
            int p = t.find(res[i], pos);
            if (p >= 0 && (rep == -1 || p < rep)) {
                rep = p;
                len = terms[i].length();
            }
        }
        if (rep >= 0) {
            int s = (rep-pre > 0) ?rep-pre :0;
            s = t.findRev(" ", s);
            if (s == -1) s = (rep-pre < 0) ?0 : rep-pre;
            int e = t.find(" ", rep+len+post);
            if (e == -1) e = t.length();
            if (lasts == -1) {
                lasts = s;
            } else if (s > laste) {
                if (out.length() == 0 && lasts > 0) out += "... ";
                out += t.mid(lasts, laste - lasts) + " ... ";
                lasts = s;
            }
            laste = e;
            pos = rep+1;
        } else {
            pos = rep;
        }
    }
    if (lasts != -1) {
        if (out.length() == 0 && lasts > 0) out += "... ";
        out += t.mid(lasts, laste - lasts) + " ... ";
    }
    for (uint i = 0; i < res.size(); ++i) {
        out.replace(res[i], "<b>\\1</b>");
    }
    if (out.length() == 0) {
        out = t.left(maxlen);
    }
    return out;
}
string
kio_strigiProtocol::Helper::mimetypeDescription(const string& mimetype) const {
    QString mt = KMimeType::mimeType(mimetype)->comment();
    if (mt != KMimeType::mimeType("")->comment()) {
        return (const char*)mt.utf8();
    }
    return mimetype;
}
kio_strigiProtocol::kio_strigiProtocol(const QCString& protocol, const QCString &pool_socket, const QCString &app_socket)
    : SlaveBase(protocol, pool_socket, app_socket), helper(new Helper()), htmlgui(helper)
{
}

kio_strigiProtocol::~kio_strigiProtocol()
{
    delete helper;
}

void kio_strigiProtocol::get(const KURL& url )
{
    QString path = url.path();
    if (path.left(1) == "/") path = path.mid(1);

    // check out what the task is
    if (path.startsWith("icon/")) {
        mimeType("image/png");
        getIcon(url.path().mid(5));
        finished();
        return;
    }

    mimeType("text/html");

    // retrieve the data from the gui object
    ostringstream out;
    map<string, string> params;
    QMap<QString, QString> q = url.queryItems();
    QMap<QString, QString>::Iterator i;
    for (i = q.begin(); i != q.end(); ++i) {
        string v = (const char*)i.data().replace('\'', "").utf8();
        params[(const char*)i.key().utf8()] = v;
    }
    htmlgui.printPage(out, path, params);

    // write the data to the client
    int size = out.str().length();
    totalSize(size);
    QCString d;
    d.setRawData(out.str().c_str(), size);
    data(d);
    d.resetRawData(out.str().c_str(), size);
    data(QByteArray()); // empty array means we're done sending the data
    finished();
}
void
kio_strigiProtocol::getIcon(const QString& url ) {
    // get the mime type
    int pos = url.find('/');
    QString mimetype;
    if (pos != -1) pos = url.find('/', pos+1);
    QString file;
    QString iconsizestring = "0";
    int iconsize = 0;
    if (pos == -1) {
        mimetype = "unknown/unknown";
        file = url;
    } else {
        mimetype = url.left(pos);
        file = url.mid(pos+1);
        pos = file.find('/');
        if (pos != -1) {
            iconsizestring = file.left(pos);
            iconsize = iconsizestring.toInt();
            if (iconsize <= 0) {
                iconsizestring = "0";
            }
        }
        file = file.mid(pos);
    }

    // first try to use PreviewJob
    dataSent = 0;
    KFileItem* fileitem = new KFileItem(file, mimetype, S_IFREG);
    KFileItemList fileitemlist;
    fileitemlist.append(fileitem);
    KIO::Job* job = KIO::filePreview(fileitemlist, iconsize, iconsize, iconsize/4, 128);
    connect(job, SIGNAL(gotPreview(const KFileItem*, const QPixmap&)),
            SLOT(slotPreview(const KFileItem*, const QPixmap&)));
    connect( job, SIGNAL( result(KIO::Job *) ),
             this, SLOT( jobFinished(KIO::Job *) ) );
    qApp->eventLoop()->enterLoop();
    if (dataSent > 0) {
        data(icondata);
        icondata.resize(0);
        data(QByteArray());
        return;
    }

    dataSent = 0;
/*    icondata.resize(0);
    dataSent = 0;
    job = KIO::get("thumbnail:"+file);
    QMap<QString, QString> meta;
    meta.insert("mimeType", mimetype);
    meta.insert("width", iconsizestring);
    meta.insert("height", iconsizestring);
    meta.insert("iconAlpha", "128");
    meta.insert("iconSize", QString::number(iconsize/4));
    job->setMetaData(meta);
    connect( job, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             this, SLOT( slotData( KIO::Job*, const QByteArray& ) ) );
    connect( job, SIGNAL( result(KIO::Job *) ),
             this, SLOT( jobFinished(KIO::Job *) ) );
    qApp->eventLoop()->enterLoop(); */

    if (dataSent > 0) {
        data(icondata);
        icondata.resize(0);
    } else {
        QString name = KMimeType::mimeType(mimetype)->icon(QString::null, -iconsize);
        name = helper->strigiiconloader.iconPath(name, KIcon::Desktop);
        QFile iconfile(name);
        iconfile.open(IO_ReadOnly);
       // kdDebug(7101) << iconsize << '-' << name << '-' << mimetype;
        data(iconfile.readAll());
    }
    data(QByteArray());
}
void
kio_strigiProtocol::slotPreview (const KFileItem *, const QPixmap &preview) {
   QBuffer buf;
    buf.open(IO_WriteOnly);
    preview.save(&buf,"PNG");
    buf.close();
    data(buf.buffer());
    dataSent = buf.buffer().size();
//    dataSent = 1;
 //   kdDebug(7101) << "image! " << dataSent;
}
void
kio_strigiProtocol::slotData(KIO::Job *job, const QByteArray &data) {
    if (dataSent == -1) return;
    if (job->error()) {
        dataSent = -1;
        return;
    }
    dataSent += data.size();
    int off = icondata.size();
    int size = data.size();
    int start = 0;
    // kio_thumbnail may return a serialized QImage which is just png
    // file with junk in front. So if this is the case we skip until the real data
    if (off == 0) {
        start = data.find('P');
        if (start < 1) {
            kdDebug(7101) << "error in reading png!";
            return;
        }
        start -= 1;
        //kdDebug(7101) << "start: " << start << " off "<<off<<" size " << size;
    }
    icondata.resize(off+size-start);
    for (int i=start; i<size; ++i) {
        icondata[i+off-start] = data[i];
    }
 //       kdDebug(7101) << "size: " << icondata.size();
}
void
kio_strigiProtocol::jobFinished( KIO::Job* job) {
    if (job->error()) {
        dataSent = -1;
        kdDebug(7101) << "error: " << job->errorText();
    } else {
 //        kdDebug(7101) << "done! " << dataSent;
    }
    qApp->eventLoop()->exitLoop();
}
void kio_strigiProtocol::mimetype(const KURL & /*url*/)
{
    mimeType("text/html");
}
void kio_strigiProtocol::listDir ( const KURL& /*url*/ ) {
    UDSEntryList entries;
    UDSEntry entry;
    UDSAtom atom;
    atom.m_uds = UDS_NAME;
    atom.m_long = 0;
    atom.m_str = "testfile";
    entry.append(atom);
    entries.append(entry);
    listEntries (entries);
}


extern "C"
{
    int kdemain(int argc, char **argv)
    {
        //KInstance instance( "kio_strigi" );
        putenv(strdup("SESSION_MANAGER="));
        KApplication::disableAutoDcopRegistration();
        KApplication app(argc, argv, "kio_strigi", false, true);
        if (argc != 4)  {
            kdError(7101) << "Usage: kio_strigi protocol domain-socket1 domain-socket2" << endl;
            exit(-1);
        }
        kio_strigiProtocol slave( "strigi", argv[2], argv[3] );
        slave.dispatchLoop();

        return 0;
    }
} 
