/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "RevisionWidget.h"
#include "sublib/ExternButton.h"
#include "svn/Revision.h"

// qt
#include <qcombobox.h>
#include <qdatetimeedit.h>
#include <qspinbox.h>
#include <qgroupbox.h>
#include <qtabwidget.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qpopupmenu.h>
#include <qwidgetstack.h>
#include <qcheckbox.h>
#include <qtoolbutton.h>


// apr
#include <apr_time.h>



// symbolic revisions
const QString workingRev   = _q("working");
const QString headRev      = _q("head");
const QString baseRev      = _q("base");
const QString committedRev = _q("committed");
const QString previousRev  = _q("previous");

// pages
const int noneId    = -1;  // in case the widget is disabled
const int symRevId  = 0;
const int numRevId  = 1;
const int dateRevId = 2;


RevisionWidget::RevisionWidget( bool allowNoRevision, const QString& types,
  const QString& symbols, RevisionProvider* prov, QWidget *parent )
  : super( parent, "RevisionWidget" ), _symRev(0), _numRev(0), _dateRev(0),
  _revpro(prov)
{
  setTitle( _q("revision: ") );
  setColumnLayout( 2, Horizontal );
  setInsideMargin(5);
  setFlat(true);

  _disable = new QCheckBox(this);
  _revBox  = new QHBox(this);
  _revBox->setSpacing(4);
  {
    _menuButton = new QToolButton(Qt::DownArrow,_revBox);
    _menu       = new QPopupMenu(_revBox);
    _stack      = new QWidgetStack(_revBox);

    for( unsigned int c = 0; c < types.length(); c++ )
    {
      addType( types.at(c) );
    }

    for( unsigned int c = 0; c < symbols.length(); c++ )
    {
      addSymbol( symbols.at(c) );
    }

    // does not work!
    //_menuButton->setPopup(_menu);
    //_menuButton->setPopupDelay(0);
    
    _stack->setMinimumWidth(_stack->width()+_stack->width()/2);
    _stack->setSizePolicy(QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Fixed));

    connect( _menu, SIGNAL(activated(int)), _stack, SLOT(raiseWidget(int)));
    connect( _menuButton, SIGNAL(clicked()), this, SLOT(popup()));

    QVBox* qvb2 = new QVBox(this);
    qvb2->layout()->setAlignment( Qt::AlignTop );
    _extern = new ExternButton(qvb2);

    if( ! _revpro )
    {
      qvb2->setHidden(true);
    }
  }

  if( ! allowNoRevision )
  {
    _disable->setChecked(true);
    _disable->hide();
  }
  else
  {
    toggled(false);
    connect( _disable, SIGNAL(toggled(bool)), SLOT(toggled(bool)) );
  }
}

RevisionWidget::~RevisionWidget()
{
}

void RevisionWidget::toggled( bool b )
{
  _revBox->setEnabled(b);
}

void RevisionWidget::popup()
{
  _menu->popup(mapToGlobal(_menuButton->pos()));
}

void RevisionWidget::addType( const QChar& c )
{
  switch( c )
  {
  case 'S':
    {
      _symRev = new QComboBox(_stack);
      _stack->addWidget(_symRev,symRevId);

      _menu->insertItem( _q("Symbolic Revision"), symRevId );
      break;
    }
  case 'N':
    {
      // todo right align
      _numRev = new QSpinBox(_stack);
      _numRev->setMaxValue( 100000000 );
      _stack->addWidget(_numRev,numRevId);

      _menu->insertItem( _q("Numeric Revision"), numRevId );
      break;
    }
  case 'D':
    {
      _dateRev = new QDateTimeEdit( QDate::currentDate(), _stack );
      _stack->addWidget(_dateRev,dateRevId);

      _menu->insertItem( _q("Date Revision"), dateRevId );
      break;
    }
  }
}

void RevisionWidget::addSymbol( const QChar& c )
{
  if( ! _symRev )
    return;

  switch( c )
  {
  case 'H':
    {
      _symRev->insertItem( headRev );
      break;
    }
  case 'B':
    {
      _symRev->insertItem( baseRev );
      break;
    }
  case 'C':
    {
      _symRev->insertItem( committedRev );
      break;
    }
  case 'P':
    {
      _symRev->insertItem( previousRev );
      break;
    }
  case 'W':
    {
      _symRev->insertItem( workingRev );
      break;
    }
  }
}


/** 
 * \brief get the selected revision.
 *
 * \return it is the callers responsibility to delete the object.
 */

svn::Revision* RevisionWidget::getRevision() const
{
  int index = _stack->id( _stack->visibleWidget() );

  if( isEnabled() == false || ! _disable->isChecked() )
  {
    index = noneId; 
  }

  switch( index )
  {
  case symRevId:
    {
      QString text = _symRev->currentText();

      if( text == headRev )
      {
        return new svn::Revision(svn::Revision_Head);
      }
      else if( text == baseRev )
      {
        return new svn::Revision(svn::Revision_Base);
      }
      else if( text == previousRev )
      {
        return new svn::Revision(svn::Revision_Previous);
      }
      else if( text == committedRev )
      {
        return new svn::Revision(svn::Revision_Committed);
      }
      else if( text == workingRev )
      {
        return new svn::Revision(svn::Revision_Working);
      }
      else
      {
        return new svn::Revision(svn::Revision_Unspecified);
      }
      break;
    }
  case numRevId:
    {
      return new svn::RevisionNumber( _numRev->value() );
      break;
    }
  case dateRevId:
    {
      // surprisingly easy... :)
      // both times are relative to the 1 January, 1970 UTC, 00:00:00
      // the only difference is time resolution:
      // apr uses microseconds and qt uses seconds
      svn::Date svndate = apr_time_from_sec( _dateRev->dateTime().toTime_t() );

      return new svn::RevisionDate(svndate);
    }
  default:
    {
      return new svn::Revision(svn::Revision_Unspecified);
    }
  }
}

void RevisionWidget::setRevision( const svn::Revision* rev )
{
  switch( rev->getKind() )
  {
  case svn::Revision_Unspecified:
    {
      // nop
      break;
    }
  case svn::Revision_Date:
    {
      svn::Date date = ((const svn::RevisionDate*)rev)->getDate();
      QDateTime qdate;
      qdate.setTime_t( apr_time_sec(date) );
      _dateRev->setDateTime(qdate);
      _stack->raiseWidget(dateRevId);
      break;
    }
  case svn::Revision_Number:
    {
      svn::Revnumber revnum = ((const svn::RevisionNumber*)rev)->getNumber();
      _numRev->setValue(revnum);
      _stack->raiseWidget(numRevId);
      _disable->setChecked(true);
      break;
    }
  case svn::Revision_Working:
    {
      _symRev->setCurrentText(workingRev);
      _stack->raiseWidget(symRevId);
      break;
    }
  case svn::Revision_Head:
    {
      _symRev->setCurrentText(headRev);
      _stack->raiseWidget(symRevId);
      break;
    }
  case svn::Revision_Base:
    {
      _symRev->setCurrentText(baseRev);
      _stack->raiseWidget(symRevId);
      break;
    }
  case svn::Revision_Committed:
    {
      _symRev->setCurrentText(committedRev);
      _stack->raiseWidget(symRevId);
      break;
    }
  case svn::Revision_Previous:
    {
      _symRev->setCurrentText(previousRev);
      _stack->raiseWidget(symRevId);
      break;
    }
  }
}
