/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "DialogPreferences.h"
#include "Config.h"
#include "Encodings.h"
#include "utility.h"
#include "ISubtitleEditor.h"

/*
 *
 */
TreeViewEncodings::TreeViewEncodings(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
:Gtk::TreeView(cobject)
{
	// get widget
	refGlade->get_widget("button-add-charset", m_buttonAddCharset);
	refGlade->get_widget("button-remove-charset", m_buttonRemoveCharset);
	refGlade->get_widget("button-up-charset", m_buttonUpCharset);
	refGlade->get_widget("button-down-charset", m_buttonDownCharset);
	refGlade->get_widget("check-used-auto-detected", m_checkUsedAutoDetected);
	
	
	// create treeview
	Column m_column;
	
	m_model = Gtk::ListStore::create(m_column);

	set_model(m_model);

	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* name = NULL;
	Gtk::CellRendererText* charset = NULL;

	// column name
	column = manage(new Gtk::TreeViewColumn(_("Name")));
	append_column(*column);
	
	name = manage(new Gtk::CellRendererText);
	column->pack_start(*name, false);
	column->add_attribute(name->property_text(), m_column.name);

	// column charset
	column = manage(new Gtk::TreeViewColumn(_("Charset")));
	append_column(*column);

	charset = manage(new Gtk::CellRendererText);
	column->pack_start(*charset, false);
	column->add_attribute(charset->property_text(), m_column.charset);


	// connect signal
	get_selection()->signal_changed().connect(
			sigc::mem_fun(*this, &TreeViewEncodings::on_selection_changed));

	m_buttonAddCharset->signal_clicked().connect(
			sigc::mem_fun(*this, &TreeViewEncodings::on_add_charset));
	
	m_buttonRemoveCharset->signal_clicked().connect(
			sigc::mem_fun(*this, &TreeViewEncodings::on_remove_charset));
	
	m_buttonUpCharset->signal_clicked().connect(
			sigc::mem_fun(*this, &TreeViewEncodings::on_up_charset));
	
	m_buttonDownCharset->signal_clicked().connect(
			sigc::mem_fun(*this, &TreeViewEncodings::on_down_charset));

	m_checkUsedAutoDetected->signal_toggled().connect(
			sigc::mem_fun(*this, &TreeViewEncodings::on_used_auto_detected_toggled));
	
	// init ui	
	Config &cfg = Config::getInstance();

	m_buttonRemoveCharset->set_sensitive(false);
	m_buttonUpCharset->set_sensitive(false);
	m_buttonDownCharset->set_sensitive(false);
	
	bool used_auto_detected = false;
	if(cfg.get_value_bool("encodings", "used-auto-detected", used_auto_detected))
	{
		m_checkUsedAutoDetected->set_active(used_auto_detected);
	}
	
	// add encodigns pref 

	std::list<Glib::ustring> list_encodings;
	if(cfg.get_value_string_list("encodings", "encodings", list_encodings))
	{
		std::list<Glib::ustring>::const_iterator it;
		for(it = list_encodings.begin(); it!=list_encodings.end(); ++it)
		{
			EncodingInfo *info= Encodings::get_from_charset(*it);
			if(info)
			{
				Gtk::TreeIter it = m_model->append();

				(*it)[m_column.name] = info->name;
				(*it)[m_column.charset] = info->charset;
			}
		}
	}
}

void TreeViewEncodings::on_add_charset()
{
	Gtk::Dialog* m_dialog = NULL;
	Gtk::TreeView *m_treeView = NULL;
	
	Glib::RefPtr<Gnome::Glade::Xml> refXml = Gnome::Glade::Xml::create(get_share_dir("dialog-encodings-chooser.glade"));
	
	refXml->get_widget("dialog-encodings-chooser", m_dialog);
	refXml->get_widget("treeviewEncodings", m_treeView);

	Column m_column;

	// create model
	Glib::RefPtr<Gtk::ListStore> m_treeViewModel = Gtk::ListStore::create(m_column);

	m_treeView->set_model(m_treeViewModel);
	// create column
	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererText* name = NULL;
	Gtk::CellRendererText* charset = NULL;

	// column name
	column = manage(new Gtk::TreeViewColumn(_("Name")));
	m_treeView->append_column(*column);
	
	name = manage(new Gtk::CellRendererText);
	column->pack_start(*name, false);
	column->add_attribute(name->property_text(), m_column.name);

	// column charset
	column = manage(new Gtk::TreeViewColumn(_("Charset")));
	m_treeView->append_column(*column);

	charset = manage(new Gtk::CellRendererText);
	column->pack_start(*charset, false);
	column->add_attribute(charset->property_text(), m_column.charset);

	// add encodings
	for(unsigned int i=0; encodings_info[i].charset != NULL; ++i)
	{
		Gtk::TreeIter it = m_treeViewModel->append();
//		(*it)[m_column.name] = encodings_info[i].name;
//		(*it)[m_column.charset] = encodings_info[i].charset;
		
		EncodingInfo *info= Encodings::get_from_charset(encodings_info[i].charset);
		if(info)
		{
			(*it)[m_column.name] = info->name;
			(*it)[m_column.charset] = info->charset;
		}
	}

	if( m_dialog->run() == Gtk::RESPONSE_OK)
	{
		Gtk::TreeIter iter = m_treeView->get_selection()->get_selected();
		
		if(iter)
		{
			Glib::ustring name = (*iter)[m_column.name];
			Glib::ustring charset = (*iter)[m_column.charset];
			// add in the list
			Gtk::TreeIter it = m_model->append();
			(*it)[m_column.name] = name;
			(*it)[m_column.charset] = charset;

			save_cfg();
		}
	}

	delete m_dialog;
}

void TreeViewEncodings::on_remove_charset()
{
	Gtk::TreeIter iter = get_selection()->get_selected();

	if(iter)
	{
		m_model->erase(iter);
		save_cfg();
	}
}

void TreeViewEncodings::on_up_charset()
{
	Gtk::TreeIter it = get_selection()->get_selected();
	if(it)
	{
		Gtk::TreePath path = m_model->get_path(it);

		if(path.prev())
		{
			Gtk::TreeIter prev = m_model->get_iter(path);
			if(prev)
			{
				m_model->move(it, prev);
				save_cfg();
			}
		}
	}
}

void TreeViewEncodings::on_down_charset()
{
	Gtk::TreeIter it = get_selection()->get_selected();
	if(it)
	{
		Gtk::TreeIter next = it;
		++next;
		if(next)
		{
			m_model->move(next, it);
			save_cfg();
		}
	}
}

void TreeViewEncodings::on_selection_changed()
{
	Gtk::TreeIter it = get_selection()->get_selected();
	
	bool state = it ? true : false;

	m_buttonRemoveCharset->set_sensitive(state);
	m_buttonUpCharset->set_sensitive(state);
	m_buttonDownCharset->set_sensitive(state);

	if(it)
	{
		// up
		Gtk::TreePath up = m_model->get_path(it);
		m_buttonUpCharset->set_sensitive(up.prev() ? true : false);
		// down
		Gtk::TreeIter down = it; ++down;
		m_buttonDownCharset->set_sensitive(down ? true : false);
	}
}

void TreeViewEncodings::on_used_auto_detected_toggled()
{
	Config::getInstance().set_value_bool("encodings", "used-auto-detected", m_checkUsedAutoDetected->get_active());
}

void TreeViewEncodings::save_cfg()
{
	Config &cfg = Config::getInstance();

	std::list<Glib::ustring> list;
		
	Column m_column;
	
	Gtk::TreeNodeChildren rows = m_model->children();

	for(Gtk::TreeIter iter = rows.begin(); iter; ++iter)
	{
		list.push_back((*iter)[m_column.charset]);
	}

	cfg.set_value_string_list("encodings", "encodings", list);
	cfg.set_value_bool("encodings", "used-auto-detected", m_checkUsedAutoDetected->get_active());

	Signal::getInstance().config_encodings_changed();
}


/*
 *	TreeView VideoPlayer
 *
 */
TreeViewVideoPlayer::TreeViewVideoPlayer(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
:Gtk::TreeView(cobject)
{
	Column m_column;
	
	m_model = Gtk::ListStore::create(m_column);

	set_model(m_model);

	//set_rules_hint(true);

	Gtk::TreeViewColumn* column = NULL;
	Gtk::CellRendererToggle* toggle = NULL;
	Gtk::CellRendererText* name = NULL;
	Gtk::CellRendererText* cmd = NULL;

	//
	column = manage(new Gtk::TreeViewColumn(_("#")));
	append_column(*column);

	toggle = manage(new Gtk::CellRendererToggle);
	column->pack_start(*toggle, false);
	column->add_attribute(toggle->property_active(), m_column.active);
	toggle->set_radio(true);
	toggle->property_activatable() = true;
	toggle->signal_toggled().connect(
			sigc::mem_fun(*this, &TreeViewVideoPlayer::on_signal_toggled));

	//
	column = manage(new Gtk::TreeViewColumn(_("Name")));
	append_column(*column);
	
	name = manage(new Gtk::CellRendererText);
	column->pack_start(*name, false);
	column->add_attribute(name->property_text(), m_column.name);

	name->property_editable() = true;
	name->signal_edited().connect(
			sigc::mem_fun(*this, &TreeViewVideoPlayer::on_signal_name_edited));

	//
	column = manage(new Gtk::TreeViewColumn(_("Command")));
	append_column(*column);

	cmd = manage(new Gtk::CellRendererText);
	column->pack_start(*cmd, false);
	column->add_attribute(cmd->property_text(), m_column.cmd);

	cmd->property_editable() = true;
	cmd->signal_edited().connect(
			sigc::mem_fun(*this, &TreeViewVideoPlayer::on_signal_cmd_edited));
	//

	// ajoute les preferences de l'utilisateur
	Config &cfg = Config::getInstance();

	std::list<Glib::ustring> list_players;
	if(cfg.get_keys("video-player", list_players))
	{
		Glib::ustring def;
		cfg.get_value_string("video-player", "default", def);

		std::list<Glib::ustring>::const_iterator it;
		for(it = list_players.begin(); it!=list_players.end(); ++it)
		{
			if((*it) != "default")
			{
				Gtk::TreeIter iter = m_model->append();

				(*iter)[m_column.name] = (*it);
				Glib::ustring cmd;
				if(cfg.get_value_string("video-player", *it, cmd))
					(*iter)[m_column.cmd] = cmd;

				if(def == (*it))
					(*iter)[m_column.active] = true;
			}
		}
	}
}

void	TreeViewVideoPlayer::on_signal_toggled(const Glib::ustring &path)
{
	Gtk::TreeIter iter = m_model->get_iter(path);
	if(iter)
	{
		// on active la ligne
		Column m_column;
		(*iter)[m_column.active] = true;

		// on sauvegarde le nouveau choix
		Config::getInstance().set_value_string("video-player", "default", (*iter)[m_column.name]);

		// on desactive les autres
		Gtk::TreeNodeChildren rows = m_model->children();
		for(Gtk::TreeIter it = rows.begin(); it; ++it)
		{
			if(it != iter)
				if((*it)[m_column.active] == true)
					(*it)[m_column.active] = false;
		}
	}
}

void TreeViewVideoPlayer::on_signal_name_edited(const Glib::ustring &path, const Glib::ustring &new_text)
{
	Gtk::TreeIter iter = m_model->get_iter(path);
	if(iter)
	{
		Column m_column;

		Glib::ustring old_text = (*iter)[m_column.name];
		if(old_text != new_text) // on evite le travail inutile
		{
			// efface l'ancienne clé
			Config &cfg = Config::getInstance();
			cfg.remove_key("video-player", old_text);

			// applique les changements
			(*iter)[m_column.name] = new_text;
			// sauvegarde le nouveau lecteur
			cfg.set_value_string("video-player", (*iter)[m_column.name], (*iter)[m_column.cmd]);
		}
	}
}

void TreeViewVideoPlayer::on_signal_cmd_edited(const Glib::ustring &path, const Glib::ustring &new_text)
{
	Gtk::TreeIter iter = m_model->get_iter(path);
	if(iter)
	{
		Column m_column;

		Glib::ustring old_text = (*iter)[m_column.cmd];
		if(old_text != new_text)
		{
			(*iter)[m_column.cmd] = new_text;
			// sauvegarde la nouvelle commande
			Config::getInstance().set_value_string("video-player", (*iter)[m_column.name], (*iter)[m_column.cmd]);
		}
	}
}

void TreeViewVideoPlayer::save_cfg()
{
	Config &cfg = Config::getInstance();
	
	// TODO clean 
	cfg.remove_group("video-player");

	Column m_column;

	Gtk::TreeNodeChildren rows = m_model->children();
	for(Gtk::TreeIter it = rows.begin(); it; ++it)
	{
		Glib::ustring name = (*it)[m_column.name];
		Glib::ustring cmd = (*it)[m_column.cmd];

		cfg.set_value_string("video-player", name, cmd);
	}

	for(Gtk::TreeIter it = rows.begin(); it; ++it)
	{
		if((*it)[m_column.active])
		{
			cfg.set_value_string("video-player", "default", (*it)[m_column.name]);
			break;
		}
	}
}


void TreeViewVideoPlayer::on_add_video_player()
{
	Column m_column;
	Gtk::TreeIter it = m_model->append();
	(*it)[m_column.name]="unknown";

	Config::getInstance().set_value_string("video-player", (*it)[m_column.name], "");

	get_selection()->select(it);
	set_cursor(m_model->get_path(it), *get_column(1), true);
}

void TreeViewVideoPlayer::on_remove_video_player()
{
	Gtk::TreeIter it = get_selection()->get_selected();
	if(it)
	{
		Column m_column;
		// on l'efface du fichier config
		Config::getInstance().remove_key("video-player", (*it)[m_column.name]);
		// s'il est activer, il faut activer un autre player
		if((*it)[m_column.active])
		{
			Gtk::TreeNodeChildren rows = m_model->children();
			Gtk::TreeIter i = rows.begin();
			if(i)
			{
				(*i)[m_column.active] = true;
			}
		}
		// et enfin on l'efface
		m_model->erase(it);
	}
}




/*
 *
 */
DialogPreferences::DialogPreferences(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
:Gtk::Dialog(cobject)
{
	// general
	refGlade->get_widget("check-maximize-window", m_checkMaximizeWindow);

	// encodings
	refGlade->get_widget_derived("treeview-encodings", m_encodingsTreeView);

	refGlade->get_widget("check-used-auto-detected", m_checkUsedAutoDetected);

	// video player
	refGlade->get_widget_derived("treeview-video-player", m_treeViewVideoPlayer);
	refGlade->get_widget("button-add-video-player", m_buttonAddVideoPlayer);
	refGlade->get_widget("button-remove-video-player", m_buttonRemoveVideoPlayer);


	// general
	m_checkMaximizeWindow->signal_toggled().connect(
			sigc::mem_fun(*this, &DialogPreferences::on_general_maximize_window_changed));

	// video preview

	m_buttonAddVideoPlayer->signal_clicked().connect(
			sigc::mem_fun(*m_treeViewVideoPlayer, &TreeViewVideoPlayer::on_add_video_player));
	
	m_buttonRemoveVideoPlayer->signal_clicked().connect(
			sigc::mem_fun(*m_treeViewVideoPlayer, &TreeViewVideoPlayer::on_remove_video_player));


	//
	Config &cfg = Config::getInstance();

	bool maximize = false;
	if(cfg.get_value_bool("general", "maximize", maximize))
	{
		m_checkMaximizeWindow->set_active(maximize);
	}
}


/*
 *
 */
void DialogPreferences::on_general_maximize_window_changed()
{
	Gtk::Window *window = dynamic_cast<Gtk::Window*>(SE::getInstance());
	
	if(m_checkMaximizeWindow->get_active())
		window->maximize();
	else
		window->unmaximize();
	save_general();
}



/*
 *
 */
void DialogPreferences::save_general()
{
	Config &cfg = Config::getInstance();

	cfg.set_value_bool("general", "maximize", m_checkMaximizeWindow->get_active());
}

/*
 *
 */
void DialogPreferences::save_video_player()
{
}

/*
 *
 */
void DialogPreferences::save_encodings()
{
	m_encodingsTreeView->save_cfg();

	Config::getInstance().set_value_bool("encodings", "used-auto-detected", m_checkUsedAutoDetected->get_active());

	Signal::getInstance().config_encodings_changed();
}
