/* $Id: keyboard.c,v 1.1.1.1 2001/07/08 13:32:58 cegger Exp $
******************************************************************************

   SVGAlib wrapper for LibGGI - keyboard handling

   Copyright (C) 1998 Marcus Sundberg	[marcus@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <internal/svgaggi.h>
#include <string.h>

#ifdef HAVE_LINUX_KEYBOARD_H /* For NR_KEYS */
#include <linux/keyboard.h>
#endif
#ifndef NR_KEYS
#define NR_KEYS		128
#endif

static unsigned char keystate[NR_KEYS];
int _ggigsw_ctrlc = GSW_CTRLC_CTRL | GSW_CTRLC_C;
int _ggigsw_keytransmask = 0;

/*
***************************************************************************
 Internal routines
***************************************************************************
*/

static int
chkscancode(int scancode)
{
	if (scancode < 0 || scancode >= NR_KEYS) {
		return 1;
	}
	return 0;
}

static void
default_keyhandler(int scancode, int newstate)
{
	if (chkscancode(scancode)) return;
	if (_ggigsw_keytransmask & TRANSLATE_CURSORKEYS)
		switch (scancode) {
		case SCANCODE_CURSORBLOCKLEFT:
			scancode = SCANCODE_CURSORLEFT;
			break;
		case SCANCODE_CURSORBLOCKRIGHT:
			scancode = SCANCODE_CURSORRIGHT;
			break;
		case SCANCODE_CURSORBLOCKUP:
			scancode = SCANCODE_CURSORUP;
			break;
		case SCANCODE_CURSORBLOCKDOWN:
			scancode = SCANCODE_CURSORDOWN;
			break;
		}
	if (_ggigsw_keytransmask & TRANSLATE_DIAGONAL) {
		switch (scancode) {
		case SCANCODE_CURSORDOWNLEFT:
			keystate[SCANCODE_CURSORDOWN] = newstate;
			keystate[SCANCODE_CURSORLEFT] = newstate;
			return;
		case SCANCODE_CURSORDOWNRIGHT:
			keystate[SCANCODE_CURSORDOWN] = newstate;
			keystate[SCANCODE_CURSORRIGHT] = newstate;
			return;
		case SCANCODE_CURSORUPLEFT:
			keystate[SCANCODE_CURSORUP] = newstate;
			keystate[SCANCODE_CURSORLEFT] = newstate;
			return;
		case SCANCODE_CURSORUPRIGHT:
			keystate[SCANCODE_CURSORUP] = newstate;
			keystate[SCANCODE_CURSORRIGHT] = newstate;
			return;
		}
	}
	if ((_ggigsw_keytransmask & TRANSLATE_KEYPADENTER)
	    && scancode == SCANCODE_KEYPADENTER) {
		scancode = SCANCODE_ENTER;
	}

	keystate[scancode] = newstate;
}

void (*_ggigsw_keyhandler) (int, int) = default_keyhandler;


/*
***************************************************************************
 Exported functions
***************************************************************************
*/

int
keyboard_init(void)
{
	keyboard_init_return_fd();

	return 0;
}

int
keyboard_init_return_fd(void)
{
	if (_ggigsw_inggi < -1) vga_init();

	/* Reset keyboard handler */
	_ggigsw_keyhandler = default_keyhandler;

	return 0; /* FIXME */
}

void
keyboard_seteventhandler(__keyboard_handler handler)
{
	_ggigsw_keyhandler = handler;
}

void
keyboard_close(void)
{
}

int
keyboard_update(void)
{
	return _ggigsw_getkeyevent(0);
}

void
keyboard_waitforupdate(void)
{
	_ggigsw_getkeyevent(1);
}

void
keyboard_setdefaulteventhandler(void)
{
	_ggigsw_keyhandler = default_keyhandler;
}

char *
keyboard_getstate(void)
{
	return keystate;
}

void
keyboard_clearstate(void)
{
	memset(keystate, 0, NR_KEYS);
	_ggigsw_ctrlc = GSW_CTRLC_CTRL | GSW_CTRLC_C;
}

void
keyboard_translatekeys(int mask)
{
	_ggigsw_keytransmask = mask;
}

int
keyboard_keypressed(int scancode)
{
	if (chkscancode(scancode)) return 0;

	return keystate[scancode];
}
