/*==================================================================
 * soundfont.c - SoundFont functions
 *
 * libInstPatch
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 *==================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include "instpatch.h"
#include "soundfont_priv.h"
#include "ippriv.h"
#include "i18n.h"

/* Some multi-thread hackery to stop segfaults with Swami between synthesis
   thread and GUI messing with generator/modulator lists.  Handled better
   in the new GObject based libInstPatch.
   See G_(UN)LOCK (instp_voice_lock) usage below. */
G_LOCK_EXTERN (instp_voice_lock);	/* defined in instpatch.c */


/* Default range value stored in 2 byte form with correct host byte order
   FIXME: Is this portable enough? */
#define DEFRANGE	(GUINT16_FROM_LE (0x7F00))

/* SF -> User unit type conversion table, comments are for NLS translators */
IPGenConv instp_gen_conv[] = {
  { NULL, NULL, NULL, NULL, 0, NULL, NULL }, /* None */
  { instp_float2int, instp_int2float, /* IPUnit_Smpls */
    instp_float2int, instp_int2float, 0,
    N_("smpls"), N_("smpls") },	/* abbreviation for "samples" */
  { instp_float2int, instp_int2float, /* IPUnit_32kSmpls */
    instp_float2int, instp_int2float, 0,
    N_("32ksmpls"), N_("32ksmpls") },/* abbreviation for 32 thousand samples */
  { instp_float2int, instp_int2float, /* IPUnit_Cent */
    instp_float2int, instp_int2float, 0,
    N_("cents"), N_("+cents") }, /* Cents (100th of a semitone) */
  { instp_hz2cents, instp_cents2hz, /* IPUnit_HzCent */
    instp_sec2tcent, instp_tcent2sec, 2,
    N_("Hz"), "X" },		/* Hertz abbreviation (cycles per second) */
  { instp_sec2tcent, instp_tcent2sec, /* IPUnit_TCent */
    instp_sec2tcent, instp_tcent2sec, 3,
    N_("sec"), "X" }, /* abbreviation for seconds (as in the time interval) */
  { instp_db2cb, instp_cb2db, instp_db2cb, instp_cb2db, 2, /* IPUnit_cB */
    N_("dB"), N_("+dB") },	/* Decibel abbreviation */
  { instp_perc2tperc, instp_tperc2perc,	/* IPUnit_Percent */
    instp_perc2tperc, instp_tperc2perc, 1,
    "%", "+%" },
  { instp_float2int, instp_int2float, /* IPUnit_Semitone */
    instp_float2int, instp_int2float, 0,
    N_("semi"), N_("+semi") },/* semitone abbreviation (1/12th of an octave) */
  { NULL, NULL, NULL, NULL, 0, N_("range"), N_("+range") } /* IPUnit_Range */
};

#define MAXNEG -32768
#define MAXPOS 32767

/* generator info */
IPGenInfo instp_gen_info[] = {
  {0, MAXPOS, 0, IPUNIT_SMPLS, N_("Start Offset")},/* StartAddrOfs */
  {MAXNEG, 0, 0, IPUNIT_SMPLS, N_("End Offset")}, /* EndAddrOfs */
  {MAXNEG, MAXPOS, 0, IPUNIT_SMPLS,
   N_("Loop Start Offset")},	/* StartLoopAddrOfs */
  {MAXNEG, MAXPOS, 0, IPUNIT_SMPLS, N_("Loop End Offset")},/* EndLoopAddrOfs */
  {0, MAXPOS, 0, IPUNIT_32KSMPLS,
   N_("Start Coarse Offset")},	/* StartAddrCoarseOfs */
  {-12000, 12000, 0, IPUNIT_CENT, N_("To Pitch")},	/* ModLFO2Pitch */
  {-12000, 12000, 0, IPUNIT_CENT, N_("To Pitch")},	/* VibLFO2Pitch */
  {-12000, 12000, 0, IPUNIT_CENT, N_("To Pitch")},	/* ModEnv2Pitch */
  {1500, 13500, 13500, IPUNIT_HZCENT, N_("Filter Cutoff")},	/* FilterFc */
  {0, 960, 0, IPUNIT_CB, N_("Filter Q")},	/* FilterQ */
  {-12000, 12000, 0, IPUNIT_CENT, N_("To Filter Cutoff")},/* ModLFO2FilterFc */
  {-12000, 12000, 0, IPUNIT_CENT, N_("To Filter Cutoff")},/* ModEnv2FilterFc */
  {MAXNEG, 0, 0, IPUNIT_32KSMPLS,
   N_("End Coarse Offset")}, /* EndAddrCoarseOfs */
  {-960, 960, 0, IPUNIT_CB, N_("To Volume")},	/* ModLFO2Vol */
  {0, 0, 0, IPUNIT_NONE, NULL},  /* Unused1 */
  {0, 1000, 0, IPUNIT_PERCENT, N_("Chorus")},	/* ChorusSend */
  {0, 1000, 0, IPUNIT_PERCENT, N_("Reverb")},	/* ReverbSend */
  {-500, 500, 0, IPUNIT_PERCENT, N_("Pan")},	/* Pan */
  {0, 0, 0, IPUNIT_NONE, NULL},	/* Unused2 */
  {0, 0, 0, IPUNIT_NONE, NULL},	/* Unused3 */
  {0, 0, 0, IPUNIT_NONE, NULL},	/* Unused4 */
  {-12000, 5000, -12000, IPUNIT_TCENT, N_("Delay")},	/* ModLFODelay */
  {-16000, 4500, 0, IPUNIT_HZCENT, N_("Frequency")},	/* ModLFOFreq */
  {-12000, 5000, -12000, IPUNIT_TCENT, N_("Delay")},	/* VibLFODelay */
  {-16000, 4500, 0, IPUNIT_HZCENT, N_("Frequency")},	/* VibLFOFreq */
  {-12000, 5000, -12000, IPUNIT_TCENT, N_("Delay")},	/* ModEnvDelay */
  {-12000, 8000, -12000, IPUNIT_TCENT, N_("Attack")},	/* ModEnvAttack */
  {-12000, 5000, -12000, IPUNIT_TCENT, N_("Hold")},	/* ModEnvHold */
  {-12000, 8000, -12000, IPUNIT_TCENT, N_("Decay")},	/* ModEnvDecay */
  {0, 1000, 0, IPUNIT_PERCENT, N_("Sustain")},	/* ModEnvSustain */
  {-12000, 8000, -12000, IPUNIT_TCENT, N_("Release")},	/* ModEnvRelease */
  {-1200, 1200, 0, IPUNIT_CENT, N_("Key to Hold")},	/* Key2ModEnvHold */
  {-1200, 1200, 0, IPUNIT_CENT, N_("Key to Decay")},	/* Key2ModEnvDecay */
  {-12000, 5000, -12000, IPUNIT_TCENT, N_("Delay")},	/* VolEnvDelay */
  {-12000, 8000, -12000, IPUNIT_TCENT, N_("Attack")},	/* VolEnvAttack */
  {-12000, 5000, -12000, IPUNIT_TCENT, N_("Hold")},	/* VolEnvHold */
  {-12000, 8000, -12000, IPUNIT_TCENT, N_("Decay")},	/* VolEnvDecay */
  {0, 1440, 0, IPUNIT_CB, N_("Sustain")},	/* VolEnvSustain */
  {-12000, 8000, -12000, IPUNIT_TCENT, N_("Release")},	/* VolEnvRelease */
  {-1200, 1200, 0, IPUNIT_CENT, N_("Key to Hold")},	/* Key2VolEnvHold */
  {-1200, 1200, 0, IPUNIT_CENT, N_("Key to Decay")},	/* Key2VolEnvDecay */
  {0, MAXPOS, 0, IPUNIT_NONE, N_("Instrument ID")}, /* Instrument */
  {0, 0, 0, IPUNIT_NONE, NULL},	        /* Reserved1 */
  {0, 127, DEFRANGE, IPUNIT_RANGE, N_("Key Range")}, /* KeyRange */
  {0, 127, DEFRANGE, IPUNIT_RANGE, N_("Velocity Range")},	/* VelRange */
  {MAXNEG, MAXPOS, 0, IPUNIT_32KSMPLS,
   N_("Loop Start Coarse Offset")}, /* StartLoopAddrCoarseOfs */
  {-1, 127, -1, IPUNIT_NONE, N_("Fixed Key")},	/* Keynum */
  {-1, 127, -1, IPUNIT_NONE, N_("Fixed Velocity")}, /* Velocity */
  {0, 1440, 0, IPUNIT_CB, N_("Attenuation")},	/* InitAttenuation */
  {0, 0, 0, IPUNIT_NONE, NULL},	                /* Reserved2 */
  {MAXNEG, MAXPOS, 0, IPUNIT_32KSMPLS,
   N_("Loop End Coarse Offset")}, /* EndLoopAddrCoarseOfs */
  {-120, 120, 0, IPUNIT_SEMITONE, N_("Coarse Tune")},	/* CourseTune */
  {-99, 99, 0, IPUNIT_CENT, N_("Fine Tune")},	/* FineTune */
  {0, MAXPOS, 0, IPUNIT_NONE, N_("Sample ID")},	/* sampleId */
  {MAXNEG, MAXPOS, 0, IPUNIT_NONE, N_("Sample Modes")},	/* SampleModes */
  {0, 0, 0, IPUNIT_NONE, NULL},	/* Reserved3 */
  {0, 1200, 100, IPUNIT_CENT, N_("Scale Tune")}, /* ScaleTuning */
  {0, 127, 0, IPUNIT_NONE, N_("Exclusive Class")}, /* ExclusiveClass */
  {-1, 127, -1, IPUNIT_NONE, N_("Override Root Key")} /* OverrideRootKey */
};


/* -- private function prototypes -- */

static int preset_qsort_compar_func (const void *a, const void *b);


/* predefined error message strings that are used multiple times */

#define IPERR_MSG_UNKNOWN_GEN_1 "Unknown IPGenType (%d)"
#define IPERR_MSG_UNKNOWN_UNIT_TYPE_1 "Invalid IPUnitType (%d)"
#define IPERR_MSG_INVALID_INFO_ID_1 "Invalid IPINFO id (%d)"


/**
 * Sets the filename of a sound font
 * @sf Sound font to set file name of
 * @file_name Path and name to set filename to
 */
void
instp_set_file_name (IPSFont *sf, const char *file_name)
{
  char *new_name = NULL;

  if (file_name) new_name = g_strdup (file_name);

  if (sf->file_name) g_free (sf->file_name);
  sf->file_name = new_name;
}

/**
 * Sets file info of a sound font object
 * @sf Sound font to set file info of
 * @file_info File info to use with the sound font. This is used
 *   directly and should not be modified or freed after calling this function.
 *
 * File handles of sound font files are kept open for sample data that
 * references the file. This function sets a sound fonts authoritive file
 * information.
 */
void
instp_set_file_info (IPSFont *sf, IPFileInfo *file_info)
{
  g_return_if_fail (sf != NULL);
  g_return_if_fail (file_info != NULL);

  instp_file_info_ref (file_info); /* add sound font item reference */
  if (sf->file) instp_file_info_unref (sf->file); /* remove ref of old file */
  sf->file = file_info;
}

/**
 * Get a sound font info string by id
 * @sf Sound font to get info from
 * @id Chunk id enum to get value of
 * Returns: Pointer to the info string (should NOT be freed) or NULL if not
 * found
 */
char *
instp_get_info (const IPSFont *sf, IPChunkType id)
{
  char *idstr;

  idstr = ipfile_chunk_enum_to_str (id);
  if (!idstr) return (NULL);
  return (instp_get_info_custom (sf, idstr));
}

/**
 * Get a sound font info string by custom 4 character id
 * @sf Sound font to get info from
 * @id 4 character chunk id string
 * Returns: Pointer to the info string (should NOT be freed) or NULL if not
 * found
 * \see instp_get_info
 */
char *
instp_get_info_custom (const IPSFont *sf, const char id[4])
{
  IPSFontInfo *info;

  g_return_val_if_fail (sf != NULL, NULL);

  info = sf->info;
  while (info)
    {
      if (strncmp (info->id, id, 4) == 0) return (info->val);
      info = instp_info_next (info);
    }

  return (NULL);
}

/**
 * Set a sound font info string by id
 * @sf Sound font to set info of
 * @id Info chunk id enum to set value of
 * @val Value to set info to
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Set sound font info.
 * Sets changed flag.
 */
int
instp_set_info (IPSFont *sf, IPChunkType id, const char *val)
{
  char *idstr;
  char *newval = NULL;
  int maxlen;

  if (id < IPINFO_FIRST || id > IPINFO_LAST
      || !(idstr = ipfile_chunk_enum_to_str (id)))
    {
      g_critical (IPERR_MSG_INVALID_INFO_ID_1, id);
      return (INSTP_FAIL);
    }

  maxlen = instp_info_max_size (id) - 1;

  if (strlen (val) > maxlen)	/* value exceeds max length? */
    {
      int retval;

      g_warning ("IPSFontInfo.val string truncated");

      newval = g_malloc (maxlen + 1);
      if (!newval) return (INSTP_FAIL);
      retval = instp_set_info_custom (sf, idstr, newval);
      g_free (newval);
      return (retval);
    }
  else return (instp_set_info_custom (sf, idstr, val));
}

/**
 * Set a sound font info string by 4 character id
 * @sf Sound font to set info of
 * @id 4 character info chunk id string to set value of
 * @val Value to set info to
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 * 
 * Set sound font info by 4 character ID.
 * Sets changed flag.
 */
int
instp_set_info_custom (IPSFont *sf, const char id[4], const char *val)
{
  IPSFontInfo *info, *lastinfo = NULL;
  char *newstr;

  g_return_val_if_fail (sf != NULL, INSTP_FAIL);

  /* see if info with specified id exists */
  info = sf->info;
  while (info)
    {
      if (strncmp (info->id, id, 4) == 0) break;
      lastinfo = info;
      info = instp_info_next (info);
    }

  if (val)			/* SET info to "val"? */
    {
      /* duplicate the value string */
      newstr = g_strdup (val);

      if (!info)		/* info with "id" does NOT exist? */
	{
	  if (!(info = instp_info_alloc ())) /* allocate new IPSFontInfo */
	    {
	      g_free (newstr);
	      return (INSTP_FAIL);
	    }
	  strncpy (info->id, id, 4); /* duplicate the IPSFontInfo ID */

	  /* add new IPSFontInfo onto end of list, or set sf->info if 1st info */
	  if (lastinfo) lastinfo->next = info;
	  else sf->info = info;
	}
      else g_free (info->val);	/* ?: info with "id" exists, free its value */

      info->val = newstr;	/* set the IPSFontInfo->val ptr to the new string */
    }
  else if (info)		/* ?: no, UNSET info */
    {
      if (sf->info == info)	/* if root info item, then update sf->info */
	sf->info = instp_info_next (info);
      instp_info_destroy (info); /* destroy the info item */
    }

  instp_item_set_root_changed_flag (INSTP_ITEM (sf), TRUE);

  return (INSTP_OK);
}

/**
 * Get maximum chunk size for INFO chunks
 * @infotype An IPINFO #IPChunkType
 * Returns: Maximum info chunk size (strlen + 1) or 0 if invalid INFO chunk type
 */
int
instp_info_max_size (IPChunkType infotype)
{
  if (infotype == IPINFO_COMMENT) /* comments can have up to 64k bytes */
    return (65536);

  if (infotype == IPINFO_VERSION /* versions take up 4 bytes */
      || infotype == IPINFO_ROM_VERSION)
    return (4);

  /* all other info types allow 256 bytes max */
  if (infotype >= IPINFO_FIRST && infotype <= IPINFO_LAST)
    return (256);

  return (0);
}

/**
 * Get the first info item in info list of a sound font
 * @sf Sound font to get info from
 * Returns: First info item in info list or NULL if no info
 */
IPSFontInfo *
instp_first_info (const IPSFont *sf)
{
  g_return_val_if_fail (sf != NULL, NULL);
  return (sf->info);
}

/**
 * Get next info item in list
 * @info Info item to get next info from
 * Returns: Next info item or NULL if no more info
 */
IPSFontInfo *
instp_info_next (const IPSFontInfo *info)
{
  g_return_val_if_fail (info != NULL, NULL);
  return (info->next);
}

/**
 * Allocate an info item
 * Returns: New info item or NULL on error
 *
 * Allocate a new info item. Only useful for manual info manipulation.
 */
IPSFontInfo *
instp_info_alloc (void)
{
  IPSFontInfo *info;

  info = g_malloc0 (sizeof (IPSFontInfo));
  return (info);
}

/**
 * Destroy an info item
 * @info Info item to destroy
 *
 * Only useful for manual info manipulation. \b NOT removed from owning
 * #IPSFont structure (if any).
 */
void
instp_info_destroy (IPSFontInfo *info)
{
  g_return_if_fail (info != NULL);
  if (info->val) g_free (info->val);
  g_free (info);
}

/**
 * Get first preset from a sound font
 * @sf Sound font to get preset from
 * Returns: First preset of sound font or NULL
 */
IPPreset *
instp_first_preset (const IPSFont *sf)
{
  g_return_val_if_fail (sf != NULL, NULL);
  return (sf->preset);
}

/**
 * Create a new preset in a sound font
 * @sf Sound font to create preset in
 * @name Name of new preset
 * @bank MIDI bank number of new preset
 * @psetnum MIDI preset number of new preset
 * Returns: New preset or NULL on error
 *
 * Create a new preset in a sound font.
 * Sets changed flag.
 */
IPPreset *
instp_create_preset (IPSFont *sf, const char *name, guint16 bank,
		     guint16 psetnum)
{
  IPPreset *pset;

  g_return_val_if_fail (sf != NULL, NULL);
  g_return_val_if_fail (name != NULL, NULL);

  pset = instp_preset_new ();
  if (!pset) return (NULL);

  if (instp_preset_set_name (pset, name) != INSTP_OK)
    {
      instp_item_destroy (INSTP_ITEM (pset));
      return (NULL);
    }

  pset->psetnum = psetnum;
  pset->bank = bank;
  instp_add_preset_sorted (sf, pset);

  return (pset);
}

/**
 * Add a preset into a sound font (sorted)
 * @sf Sound font to add preset to
 * @preset Preset to add
 *
 * Adds an existing preset to a sound font sorted by its bank:psetnum,
 * assumes preset list is already sorted
 * Sets changed flag.
 */
void
instp_add_preset_sorted (IPSFont *sf, IPPreset *preset)
{
  IPPreset *sibling;

  g_return_if_fail (sf != NULL);
  g_return_if_fail (preset != NULL);

  sibling = sf->preset;
  while (sibling)		/* loop over presets to find insert location */
    {
      if (instp_preset_compare (preset, sibling) < 0) break;
      sibling = instp_preset_next (sibling);
    }

  instp_item_insert_before (INSTP_ITEM (sf), INSTP_ITEM (preset),
			    INSTP_ITEM (sibling));
}

/**
 * Insert a preset into a sound font
 * @sf Sound font to insert preset into
 * @preset Preset to insert
 * @pos Index in list to insert preset into (0 = first, < 0 last)
 * \see instp_add_preset_sorted
 * 
 * Inserts a preset into a sound fonts preset list at a given index,
 * use #instp_add_preset_sorted instead to keep the preset list sorted
 * Sets changed flag.
 */
void
instp_insert_preset (IPSFont *sf, IPPreset *preset, int pos)
{
  g_return_if_fail (sf != NULL);
  g_return_if_fail (preset != NULL);

  instp_item_insert (INSTP_ITEM (sf), INSTP_ITEM (preset), pos);
}

/**
 * Get first instrument from a sound font
 * @sf Sound font to get instrument from
 * Returns: First instrument in sound font's instrument list or NULL if none
 */
IPInst *
instp_first_inst (const IPSFont *sf)
{
  g_return_val_if_fail (sf != NULL, NULL);
  return (sf->inst);
}

/**
 * Create a new instrument in a sound font
 * @sf Sound font to create instrument in
 * @name Name of new instrument
 * Returns: New instrument or NULL on error
 *
 * Create a new preset in a sound font.
 * Sets changed flag.
 */
IPInst *
instp_create_inst (IPSFont *sf, const char *name)
{
  IPInst *inst;

  g_return_val_if_fail (sf != NULL, NULL);
  g_return_val_if_fail (name != NULL, NULL);

  inst = instp_inst_new ();
  if (!inst) return (NULL);

  if (instp_inst_set_name (inst, name) != INSTP_OK)
    {
      instp_item_destroy (INSTP_ITEM (inst));
      return (NULL);
    }

  instp_add_inst (sf, inst);

  return (inst);
}

/**
 * Add an instrument to a sound font
 * @sf Sound font to add instrument to
 * @inst Instrument to add
 *
 * Appends an instrument to a sound font's instrument list (not sorted)
 * Sets changed flag.
 */
void
instp_add_inst (IPSFont *sf, IPInst *inst)
{
  g_return_if_fail (sf != NULL);
  g_return_if_fail (inst != NULL);
  instp_item_insert_before (INSTP_ITEM (sf), INSTP_ITEM (inst), NULL);
}

/**
 * Insert an instrument into a sound font
 * @sf Sound font to insert instrument into
 * @inst Instrument to insert
 * @pos Index in instrument list to insert at (0 = first, < 0 last)
 *
 * Inserts an instrument in a sound font's instrument list at the given
 * position.
 * Sets changed flag.
 */
void
instp_insert_inst (IPSFont *sf, IPInst *inst, int pos)
{
  g_return_if_fail (sf != NULL);
  g_return_if_fail (inst != NULL);
  instp_item_insert (INSTP_ITEM (sf), INSTP_ITEM (inst), pos);
}

/**
 * Get first sample from sound font
 * @sf Sound font to get sample from
 * Returns: First sample in sound font's sample list or NULL if none
 */
IPSample *
instp_first_sample (const IPSFont *sf)
{
  g_return_val_if_fail (sf != NULL, NULL);
  return (sf->sample);
}

/**
 * Create a sample in a sound font
 * @sf Sound font to create sample in
 * @name Name of new sample
 * Returns: New sample or NULL on error
 *
 * Create a new sample in a sound font.
 * Sets changed flag.
 */
IPSample *
instp_create_sample (IPSFont *sf, const char *name)
{
  IPSample *sample;

  g_return_val_if_fail (sf != NULL, NULL);
  g_return_val_if_fail (sample != NULL, NULL);

  sample = instp_sample_new ();
  if (!sample) return (NULL);

  if (instp_sample_set_name (sample, name) != INSTP_OK)
    {
      instp_item_destroy (INSTP_ITEM (sample));
      return (NULL);
    }

  instp_add_sample (sf, sample);

  return (sample);
}

/**
 * Add a sample to a sound font
 * @sf Sound font to add sample to
 * @sample Sample to add
 *
 * Append a sample to a sound font.
 * Sets changed flag.
 */
void
instp_add_sample (IPSFont *sf, IPSample *sample)
{
  g_return_if_fail (sf != NULL);
  g_return_if_fail (sample != NULL);
  instp_item_insert (INSTP_ITEM (sf), INSTP_ITEM (sample), -1);
}

/**
 * Insert a sample into a sound font
 * @sf Sound font to insert sample into
 * @sample Sample to insert
 * @pos Index in sound font's sample list (0 = first, < 0 = last)
 *
 * Insert a sample into a sound font.
 * Sets changed flag.
 */
void
instp_insert_sample (IPSFont *sf, IPSample *sample, int pos)
{
  g_return_if_fail (sf != NULL);
  g_return_if_fail (sample != NULL);
  instp_item_insert (INSTP_ITEM (sf), INSTP_ITEM (sample), pos);
}


/* temp sorting structure used in instp_find_free_preset */
typedef struct _TempPresetSort
{
  guint16 bank;
  guint16 psetnum;
} TempPresetSort;

/**
 * Find first unused MIDI bank:preset numbers in a sound font
 * @sf Sound font to search in
 * @bank INPUT: Pointer to bank number to start search on.
 *   OUTPUT: Pointer to store MIDI bank number of free preset in.
 * @psetnum OUTPUT: Pointer to store MIDI preset number in.
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise (memory alloc error, etc)
 *
 * Searches through a sound font's preset list for the first unused MIDI
 * bank:preset numbers which are returned in \b bank and \b psetnum
 * respectively. The bank parameter is also used as input which is the initial
 * bank to start the search on. Since preset list may not be sorted, a
 * temporary sorted array is used.
 */
int
instp_find_free_preset (const IPSFont *sf, int *bank, int *psetnum)
{
  TempPresetSort *sort_array;	/* array of bank:psetnum pairs to sort */
  TempPresetSort *p;
  IPPreset *pset;
  int b, n;			/* Stores current bank and preset number */
  int i, count;

  g_return_val_if_fail (sf != NULL, INSTP_FAIL);

  /* count number of presets */
  count = instp_item_count (INSTP_ITEM (sf->preset));
  if (count == 0)
    {
      *bank = 0;
      *psetnum = 0;
      return (INSTP_OK);
    }

  sort_array = g_malloc (sizeof (TempPresetSort) * count);
  if (!sort_array) return (INSTP_FAIL);

  /* fill array with bank and preset numbers */
  pset = sf->preset;
  for (i = 0; i < count; i++)
    {
      sort_array[i].bank = pset->bank;
      sort_array[i].psetnum = pset->psetnum;

      pset = instp_preset_next (pset);
    }

  qsort (sort_array, count, sizeof (TempPresetSort), /* sort array */
	 preset_qsort_compar_func);

  b = *bank;
  n = 0;

  /* loop through sorted array of bank:psetnums */
  for (i = 0; i < count; i++)
    {
      p = &sort_array[i];

      if (p->bank > b || (p->bank == b && p->psetnum > n)) break;
      if (p->bank >= b)
	{
	  if (++n > 127)
	    {
	      n = 0;
	      b++;
	    }
	}
    }
  *bank = b;
  *psetnum = n;

  g_free (sort_array);		/* free the temp sort array */
  return (INSTP_OK);
}

/* function used to do a temporary sort on preset list for
   instp_find_free_preset (see: `man qsort') */
static int
preset_qsort_compar_func (const void *a, const void *b)
{
  TempPresetSort *at = (TempPresetSort *)a, *bt = (TempPresetSort *)b;
  gint32 aval, bval;

  aval = ((gint32)(at->bank) << 16) | at->psetnum;
  bval = ((gint32)(bt->bank) << 16) | bt->psetnum;

  return (aval - bval);
}

/**
 * Find a preset in a sound font
 * @sf Sound font to search in
 * @name Name of preset to find or NULL to match any name
 * @bank MIDI bank number of preset to search for or -1 to not search by
 *   MIDI bank:preset numbers
 * @psetnum MIDI preset number of preset to search for, only used
 *   if \b bank is 0-128
 * @exclude A preset to exclude from the search or NULL
 * Returns: The matching preset or NULL if not found
 *
 * Find a preset by name or bank:preset MIDI numbers. If preset name and
 * bank:psetnum are specified then match for either condition.
 */
IPPreset *
instp_find_preset (const IPSFont *sf, const char *name, int bank,
		   int psetnum, const IPPreset *exclude)
{
  IPPreset *pset;
  gboolean bynum = FALSE;

  g_return_val_if_fail (sf != NULL, NULL);

  /* if bank and psetnum are valid, then search by number */
  if (bank >= 0 && bank <= 128 && psetnum >= 0 && psetnum < 128)
    bynum = TRUE;

  pset = sf->preset;
  while (pset)
    {
      if (pset != exclude	/* if exclude is NULL it will never == pset */
	  && ((bynum && pset->bank == bank && pset->psetnum == psetnum)
	      || (name && strcmp (pset->name, name) == 0)))
	return (pset);
      pset = instp_preset_next (pset);
    }
  return (NULL);
}

/**
 * Find an instrument by name in a sound font
 * @sf Sound font to search in
 * @name Name of Instrument to find
 * @exclude An instrument to exclude from the search or NULL
 * Returns: The matching instrument or NULL if not found
 */
IPInst *
instp_find_inst (const IPSFont *sf, const char *name, const IPInst *exclude)
{
  IPInst *inst;

  g_return_val_if_fail (sf != NULL, NULL);
  g_return_val_if_fail (name != NULL, NULL);

  inst = sf->inst;
  while (inst)
    {
      if (inst != exclude && strcmp (inst->name, name) == 0)
	return (inst);
      inst = instp_inst_next (inst);
    }
  return (NULL);
}

/**
 * Find a sample by name in a sound font
 * @sf Sound font to search in
 * @name Name of sample to find
 * @exclude A sample to exclude from the search or NULL
 * Returns: The matching sample or NULL if not found
 */
IPSample *
instp_find_sample (const IPSFont *sf, const char *name, const IPSample *exclude)
{
  IPSample *sample;

  g_return_val_if_fail (sf != NULL, NULL);
  g_return_val_if_fail (name != NULL, NULL);

  sample = sf->sample;
  while (sample)
    {
      if (sample != exclude && strcmp (sample->name, name) == 0)
	return (sample);
      sample = instp_sample_next (sample);
    }
  return (NULL);
}

/**
 * Create a new sound font file info structure
 * Returns: New initialized file info structure or NULL on error
 */
IPFileInfo *
instp_file_info_new (void)
{
  IPFileInfo *file_info;

  file_info = g_malloc0 (sizeof (IPFileInfo));
  if (file_info) file_info->fd = -1;
  return (file_info);
}

/**
 * Increment a sound font file info reference count
 * @info The sound font file info to reference
 */
void
instp_file_info_ref (IPFileInfo *info)
{
  g_return_if_fail (info != NULL);

  info->ref_count++;
}

/**
 * Decrement a sound font file info reference count
 * @info The sound font file info to un-reference
 */
void
instp_file_info_unref (IPFileInfo *info)
{
  g_return_if_fail (info != NULL);
  g_return_if_fail (info->ref_count > 0);

  if (!(--info->ref_count))
    {
      if (info->fd != -1) close (info->fd);
      g_free (info);
    }
}

/* #IPItem_InitFunc for #IPSFont items */
int
instp_sfont_init_func (IPItem *item)
{
  IPSFont *sf;

  sf = INSTP_SFONT (item);

  if (!(sf->file = instp_file_info_new ()))
    return (INSTP_FAIL);

  instp_file_info_ref (sf->file);

  sf->version.major = 2;
  sf->version.minor = 1;

  /* set sound font name to default INAM string */
  instp_set_info (sf, IPINFO_NAME, _(INSTP_DEFAULT_INAM));

  /* set sound font sound engine to default ISNG string */
  instp_set_info (sf, IPINFO_ENGINE, INSTP_DEFAULT_ISNG);

  /* set sound font software string to default ISFT string */
  instp_set_info (sf, IPINFO_SOFTWARE, INSTP_DEFAULT_ISFT);

  sf->flag_changed = FALSE;
  sf->flag_saved = FALSE;

  return (INSTP_OK);
}

/* #IPItem_DoneFunc for #IPSFont items */
void
instp_sfont_done_func (IPItem *item)
{
  IPSFont *sf;
  IPSFontInfo *info, *temp;

  g_return_if_fail (item != NULL);
  g_return_if_fail (INSTP_IS_SFONT (item));

  sf = INSTP_SFONT (item);

  info = sf->info;
  while (info)
    {
      temp = info;
      info = instp_info_next (info);
      instp_info_destroy (temp);
    }
  sf->info = NULL;

  instp_file_info_unref (sf->file);
  sf->file = NULL;
}

/* #IPItem_DestroyFunc for #IPSFont items */
void
instp_sfont_destroy_func (IPItem *item)
{
  IPSFont *sf;
  IPItem *p, *temp;

  g_return_if_fail (item != NULL);
  g_return_if_fail (INSTP_IS_SFONT (item));

  sf = INSTP_SFONT (item);

  /* unlink all the presets */
  p = INSTP_ITEM (sf->preset);
  while (p)
    {
      temp = p;
      p = instp_item_next (p);
      instp_item_unlink (p);
    }

  /* unlink all the instruments */
  p = INSTP_ITEM (sf->inst);
  while (p)
    {
      temp = p;
      p = instp_item_next (p);
      instp_item_unlink (p);
    }

  /* unlink all the samples */
  p = INSTP_ITEM (sf->sample);
  while (p)			/* loop over samples */
    {
      temp = p;
      p = instp_item_next (p);
      instp_item_unlink (p);
    }

  instp_item_unlink (item); /* unlink the sound font */
}

/* #IPItem_DoneFunc for #IPPreset items */
void
instp_preset_done_func (IPItem *item)
{
  g_return_if_fail (item != NULL);
  g_return_if_fail (INSTP_IS_PRESET (item));

  if (INSTP_PRESET (item)->name) /* free preset name if any */
    g_free (INSTP_PRESET (item)->name);
}

/* #IPItem_DestroyFunc for #IPPreset items */
void
instp_preset_destroy_func (IPItem *item)
{
  IPPreset *preset;
  IPZone *zone, *temp;

  g_return_if_fail (item != NULL);
  g_return_if_fail (INSTP_IS_PRESET (item));

  preset = INSTP_PRESET (item);

  /* unlink zones */
  zone = instp_preset_first_zone (preset);
  while (zone)
    {
      temp = zone;
      zone = instp_zone_next (zone);
      instp_item_unlink (INSTP_ITEM (temp)); /* unlink the zone */
    }

  instp_item_unlink (item);	/* unlink the preset */
}

/* #IPItem_DuplicateFunc for #IPPreset items */
IPItem *
instp_preset_duplicate_func (const IPItem *item)
{
  IPPreset *preset, *dup;
  IPZone *dupz;
  IPZone *zone;

  g_return_val_if_fail (item != NULL, NULL);
  g_return_val_if_fail (INSTP_IS_PRESET (item), NULL);

  preset = INSTP_PRESET (item);

  dup = instp_preset_new ();
  if (!dup) return (NULL);

  dup->sfitem.userptr = preset->sfitem.userptr;

  /* set the duplicate preset name */
  if (instp_preset_set_name (dup, preset->name) != INSTP_OK)
    {
      instp_item_destroy (INSTP_ITEM (dup));
      return (NULL);
    }

  dup->psetnum = preset->psetnum;
  dup->bank = preset->bank;
  dup->library = preset->library;
  dup->genre = preset->genre;
  dup->morphology = preset->morphology;

  zone = instp_preset_first_zone (preset);
  while (zone)			/* loop over preset zones and duplicate them */
    {
      dupz = instp_zone_duplicate (zone);
      if (!dupz)		/* zone duplicate could fail */
	{
	  instp_item_destroy (INSTP_ITEM (dup));
	  return (NULL);
	}

      instp_preset_add_zone (dup, dupz);
      zone = instp_zone_next (zone);
    }

  return (INSTP_ITEM (dup));
}

/**
 * Preset comparison function for sorting
 * @p1 First preset in comparison
 * @p2 Second preset in comparison
 * Returns: Comparison result that is less than, equal to, or greater than zero
 * if p1 is found, respectively, to be less than, to match, or be greater
 * than p2
 *
 * Compare two presets by their MIDI bank:preset numbers
 */
int
instp_preset_compare (const IPPreset *p1, const IPPreset *p2)
{
  gint32 aval, bval;

  aval = ((gint32)(p1->bank) << 16) | p1->psetnum;
  bval = ((gint32)(p2->bank) << 16) | p2->psetnum;

  return (aval - bval);
}

/**
 * Set a presets name
 * @preset Preset to set name of
 * @name String to set presets name to
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Set a preset's name.
 * Sets changed flag.
 */
int
instp_preset_set_name (IPPreset *preset, const char *name)
{
  char *dupname;

  g_return_val_if_fail (preset != NULL, INSTP_FAIL);
  g_return_val_if_fail (name != NULL, INSTP_FAIL);

  dupname = g_strdup (name);

  if (preset->name) g_free (preset->name);
  preset->name = dupname;

  instp_item_set_root_changed_flag (INSTP_ITEM (preset), TRUE);

  return (INSTP_OK);
}

/**
 * Get a presets first zone
 * @preset Preset to get zone from
 * Returns: First zone in preset's zone list or NULL if none
 */
IPZone *
instp_preset_first_zone (const IPPreset *preset)
{
  g_return_val_if_fail (preset != NULL, NULL);
  return (preset->zone);
}

/**
 * Create a zone in a preset
 * @preset Preset to create zone in
 * @zone_inst Pointer to new zone's referenced #IPInst or NULL for
 * global zone
 * Returns: The new zone or NULL on error
 *
 * Create a new zone in a preset.
 * Sets changed flag.
 */
IPZone *
instp_preset_create_zone (IPPreset *preset, const IPInst *zone_inst)
{
  IPZone *zone;

  g_return_val_if_fail (preset != NULL, NULL);

  zone = instp_zone_new ();
  if (!zone) return (NULL);
  instp_zone_set_refitem (zone, (IPItem *)zone_inst);
  instp_preset_add_zone (preset, zone);

  return (zone);
}

/**
 * Add a zone to a preset
 * @preset Preset to add zone to
 * @zone Zone to add (append) to preset's zone list
 *
 * Adds a zone to a preset.
 * Sets changed flag.
 */
void
instp_preset_add_zone (IPPreset *preset, IPZone *zone)
{
  instp_item_append (INSTP_ITEM (preset), INSTP_ITEM (zone));
}

/**
 * Insert zone in preset
 * @preset Preset to insert zone into
 * @zone Zone to insert
 * @pos Index position in preset's zone list for insert (0 = first,
 * < 0 = last)
 *
 * Inserts a zone in a preset at a given position.
 * Sets changed flag.
 */
void
instp_preset_insert_zone (IPPreset *preset, IPZone *zone, int pos)
{
  instp_item_insert (INSTP_ITEM (preset), INSTP_ITEM (zone), pos);
}

/* -------------------------- */
/* ---- IPInst functions ---- */
/* -------------------------- */

/* #IPItem_DoneFunc for #IPInst items */
void
instp_inst_done_func (IPItem *item)
{
  g_return_if_fail (item != NULL);
  g_return_if_fail (INSTP_IS_INST (item));

  if (INSTP_INST (item)->name)	/* free instrument name if any */
    g_free (INSTP_INST (item)->name);
}

/* #IPItem_DestroyFunc for #IPInst items */
void
instp_inst_destroy_func (IPItem *item)
{
  IPInst *inst;
  IPZone *zone, *temp;

  g_return_if_fail (item != NULL);
  g_return_if_fail (INSTP_IS_INST (item));

  inst = INSTP_INST (item);

  /* unlink zones */
  zone = instp_inst_first_zone (inst);
  while (zone)
    {
      temp = zone;
      zone = instp_zone_next (zone);
      instp_item_unlink (INSTP_ITEM (temp)); /* unlink zone */
    }

  instp_item_unlink (item);	/* unlink instrument */
}

/* #IPItem_DuplicateFunc for #IPInst items */
IPItem *
instp_inst_duplicate_func (const IPItem *item)
{
  IPInst *inst, *dup;
  IPZone *dupz;
  IPZone *zone;

  g_return_val_if_fail (item != NULL, NULL);
  g_return_val_if_fail (INSTP_IS_INST (item), NULL);

  inst = INSTP_INST (item);

  dup = instp_inst_new ();
  if (!dup) return (NULL);

  dup->sfitem.userptr = inst->sfitem.userptr;

  if (instp_inst_set_name (dup, inst->name) != INSTP_OK)
    {
      instp_item_destroy (INSTP_ITEM (dup));
      return (NULL);
    }

  zone = instp_inst_first_zone (inst);
  while (zone)			/* loop over inst zones and copy them */
    {
      dupz = instp_zone_duplicate (INSTP_ITEM (zone));
      if (!dupz)		/* zone duplicate could fail */
	{
	  instp_item_destroy (INSTP_ITEM (dup));
	  return (NULL);
	}

      instp_inst_add_zone (dup, dupz);
      zone = instp_zone_next (zone);
    }

  return (INSTP_ITEM (dup));
}

/**
 * Set an instrument's name
 * @inst Instrument to set name of
 * @name String to set instrument's name to
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Set an instrument's name.
 * Sets changed flag.
 */
int
instp_inst_set_name (IPInst *inst, const char *name)
{
  char *dupname;

  g_return_val_if_fail (inst != NULL, INSTP_FAIL);
  g_return_val_if_fail (name != NULL, INSTP_FAIL);

  dupname = g_strdup (name);

  if (inst->name) g_free (inst->name);
  inst->name = dupname;

  instp_item_set_root_changed_flag (INSTP_ITEM (inst), TRUE);

  return (INSTP_OK);
}

/**
 * Get the first zone from an instrument
 * @inst Instrument to get zone from
 * Returns: The first zone in the instrument's zone list or NULL if none
 */
IPZone *
instp_inst_first_zone (const IPInst *inst)
{
  g_return_val_if_fail (inst != NULL, NULL);
  return (inst->zone);
}

/**
 * Create a zone in an instrument
 * @inst Instrument to create zone in
 * @zone_sample Pointer to the new zone's referenced #IPSample or NULL
 * for global zone
 *
 * Create a zone in an instrument.
 * Sets changed flag.
 */
IPZone *
instp_inst_create_zone (IPInst *inst, const IPSample *zone_sample)
{
  IPZone *zone;

  g_return_val_if_fail (inst != NULL, NULL);

  zone = instp_zone_new ();
  if (!zone) return (NULL);
  instp_zone_set_refitem (zone, (IPItem *)zone_sample);
  instp_inst_add_zone (inst, zone);

  return (zone);
}

/**
 * Add a zone to an instrument
 * @inst Instrument to add zone to
 * @zone Zone to add (append)
 *
 * Append a zone to an instrument.
 * Sets changed flag.
 */
void
instp_inst_add_zone (IPInst *inst, IPZone *zone)
{
  instp_item_append (INSTP_ITEM (inst), INSTP_ITEM (zone));
}

/**
 * Insert a zone into an instrument
 * @inst Instrument to insert zone into
 * @zone Zone to insert
 * @pos Index into instrument's zone list to insert at
 * (0 = first, < 0 = last)
 *
 * Insert a zone in an instrument at the given position.
 * Sets changed flag.
 */
void
instp_inst_insert_zone (IPInst *inst, IPZone *zone, int pos)
{
  instp_item_insert (INSTP_ITEM (inst), INSTP_ITEM (zone), pos);
}

/* ---------------------------- */
/* ---- IPSample functions ---- */
/* ---------------------------- */

/* #IPItem_DoneFunc for #IPSample items */
void
instp_sample_done_func (IPItem *item)
{
  g_return_if_fail (item != NULL);
  g_return_if_fail (INSTP_IS_SAMPLE (item));

  if (INSTP_SAMPLE (item)->name) /* free sample name if any */
    g_free (INSTP_SAMPLE (item)->name);
}

/* #IPItem_DestroyFunc for #IPSample items */
void
instp_sample_destroy_func (IPItem *item)
{
  IPSample *sample;

  g_return_if_fail (item != NULL);
  g_return_if_fail (INSTP_IS_SAMPLE (item));

  sample = INSTP_SAMPLE (item);

  if (sample->sampledata)	/* unref sample's data */
    instp_item_unref (INSTP_ITEM (sample->sampledata));
  sample->sampledata = NULL;

  instp_item_unlink (item);	/* unlink sample */
}

/* #IPItem_DuplicateFunc for #IPSample items */
IPItem *
instp_sample_duplicate_func (const IPItem *item)
{
  IPSample *sample, *dup;

  g_return_val_if_fail (item != NULL, NULL);
  g_return_val_if_fail (INSTP_IS_SAMPLE (item), NULL);

  sample = INSTP_SAMPLE (item);

  dup = instp_sample_new ();
  if (!dup) return (NULL);

  dup->sfitem.userptr = sample->sfitem.userptr;

  if (instp_sample_set_name (dup, sample->name) != INSTP_OK)
    {
      instp_item_destroy (INSTP_ITEM (dup));
      return (NULL);
    }

  dup->loopstart = sample->loopstart;
  dup->loopend = sample->loopend;
  dup->samplerate = sample->samplerate;
  dup->origpitch = sample->origpitch;
  dup->pitchadj = sample->pitchadj;
  dup->sampletype = sample->sampletype;

  instp_sample_set_sample_data (dup, sample->sampledata);

  return (INSTP_ITEM (dup));
}

/**
 * Set a sample's name
 * @sample Sample to set name of
 * @name String to set sample name to
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 *
 * Set a sample's name.
 * Sets changed flag.
 */
int
instp_sample_set_name (IPSample *sample, const char *name)
{
  char *dupname;

  g_return_val_if_fail (sample != NULL, INSTP_FAIL);
  g_return_val_if_fail (name != NULL, INSTP_FAIL);

  dupname = g_strdup (name);

  if (sample->name) g_free (sample->name);
  sample->name = dupname;

  instp_item_set_root_changed_flag (INSTP_ITEM (sample), TRUE);

  return (INSTP_OK);
}

/**
 * Set a sample's referenced sample data object
 * @sample Sample to set sample data object reference in.
 * @sampledata Sample data object to reference.
 */
void
instp_sample_set_sample_data (IPSample *sample, IPSampleData *sampledata)
{
  g_return_if_fail (sample != NULL);
  g_return_if_fail (sampledata != NULL);

  if (sampledata) instp_item_ref (INSTP_ITEM (sampledata));
  if (sample->sampledata)
    instp_item_unref (INSTP_ITEM (sample->sampledata));

  sample->sampledata = sampledata;
}

/**
 * Get the sample data object of a sample
 * @sample Sample to get the referenced sample data object from.
 * Returns: Sample data object of sample or NULL if none.
 */
IPSampleData *
instp_sample_get_sample_data (IPSample *sample)
{
  g_return_val_if_fail (sample != NULL, NULL);

  return (sample->sampledata);
}

/**
 * Set the sample data reference of a sample object to blank data
 * @sample Sample to set to blank sample data
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
instp_sample_set_blank_data (IPSample *sample)
{
  IPSampleData *sampledata;

  /* fetch the blank sample data object */
  sampledata = instp_sample_data_get_blank ();
  if (!sampledata) return (INSTP_FAIL);

  instp_sample_set_sample_data (sample, sampledata);

  sample->loopstart = 8;
  sample->loopend = 40;
  sample->samplerate = 44100;
  sample->origpitch = 60;
  sample->pitchadj = 0;
  sample->sampletype = 0;
  sample->linked = NULL;

  return (INSTP_OK);
}

/**
 * Get the size of a sample
 * @sample Sample to get size of
 * Returns: Size of sample data (in samples) or -1 on error
 *
 * Sample should have been assigned a sample data object.
 */
int
instp_sample_get_size (IPSample *sample)
{
  g_return_val_if_fail (sample != NULL, -1);
  g_return_val_if_fail (sample->sampledata != NULL, -1);

  return (sample->sampledata->size);
}


/* ----------------------------- */
/* --- Sample data functions --- */
/* ----------------------------- */

/*
 * Sample data item is used differently than other items. An instance of the
 * sample data item is never explicity destroyed. Rather it exists until there
 * are no more sample references. This is the reason for the hack to not count
 * the references in the instp_sample_data_list, as this would cause the data
 * to never be destroyed.
 */

/* #IPItem_InitFunc for #IPSampleData items */
int
instp_sample_data_init_func (IPItem *item)
{
  g_return_val_if_fail (item != NULL, INSTP_FAIL);
  g_return_val_if_fail (INSTP_IS_SAMPLE_DATA (item), INSTP_FAIL);

  /* manually prepend the new sample data onto the sample data list to avoid
     ref counting by the list */
  if (instp_sample_data_list) INSTP_ITEM (instp_sample_data_list)->prev = item;
  item->next = INSTP_ITEM (instp_sample_data_list);
  instp_sample_data_list = INSTP_SAMPLE_DATA (item);

  return (INSTP_OK);
}

/* #IPItem_DoneFunc for #IPSampleData items */
void
instp_sample_data_done_func (IPItem *item)
{
  IPSampleData *sampledata;
  IPSampleStore *store, *temp;

  g_return_if_fail (item != NULL);
  g_return_if_fail (INSTP_IS_SAMPLE_DATA (item));

  sampledata = INSTP_SAMPLE_DATA (item);

  store = instp_sample_data_first_store (sampledata);
  while (store)
    {
      temp = store;
      store = instp_sample_store_next (store);
      instp_sample_store_destroy (sampledata, store);
    }

  /* manually unlink the sample data item from the sample data list to avoid
     unrefing by list */
  if (item->next) item->next->prev = item->prev;
  if (item->prev) item->prev->next = item->next;
  if (instp_sample_data_list == INSTP_SAMPLE_DATA (item))
    instp_sample_data_list = INSTP_SAMPLE_DATA (item->next);
  item->next = item->prev = NULL;
}

/**
 * Set the size of a sample data object
 * @sampledata Sample data object to set size of
 * @size Size (in samples) to set sample data object to.
 *
 * Size of sample data object should not have been set before (i.e. only new
 * sample data objects should have their size set).
 */
void
instp_sample_data_set_size (IPSampleData *sampledata, guint size)
{
  g_return_if_fail (sampledata != NULL);
  g_return_if_fail (sampledata->size == 0);

  sampledata->size = size;
}

/**
 * Get the first sample data store in a sample data object
 * @sampledata Sample data item
 * Returns: First sample data store structure or NULL if none
 */
IPSampleStore *
instp_sample_data_first_store (IPSampleData *sampledata)
{
  g_return_val_if_fail (sampledata != NULL, NULL);
  return (sampledata->storage);
}

/**
 * Find a sample store in a sample data object by a criteria
 * @sampledata Sample data object
 * @method_type Sample store method type to find (0 or
 *   IPSAMPLE_METHOD_NONE as a wildcard).
 * @find_flags Criteria for finding a sample store
 *   (IPSAMPLE_STORE_FIND_* flags).
 * Returns: A sample store matching the criteria or NULL if no match (there
 *   might not be any stores in sampledata).
 */
IPSampleStore *
instp_sample_data_find_store (IPSampleData *sampledata,
			      int method_type, int find_flags)
{
  IPSampleStore *store, *match = NULL;

  g_return_val_if_fail (sampledata != NULL, NULL);

  store = sampledata->storage;
  while (store)
    {
      /* store matches method_type, readable and writable criteria? */
      if ((method_type == 0 || method_type == store->method->type)
	  && (!(find_flags & IPSAMPLE_STORE_FIND_READABLE)
	   || IPSAMPLE_STORE_IS_READABLE (store))
	  && (!(find_flags & IPSAMPLE_STORE_FIND_WRITABLE)
	   || IPSAMPLE_STORE_IS_WRITABLE (store)))
	{
	  if (find_flags & IPSAMPLE_STORE_FIND_FASTEST)
	    {			/* find fastest matching */
	      if (match == NULL || store->method->speed_rating
		  > match->method->speed_rating)
		match = store;
	    }
	  else			/* not looking for fastest, found match */
	    {
	      match = store;
	      break;
	    }
	}
      store = instp_sample_store_next (store);
    }

  return (match);
}

/**
 * Get blank sample data object
 * Returns: The blank sample data object or NULL on error (out of memory)
 *
 * Return's a sample data structure with the minimum amount of data
 * which is blank. Only creates it on the first call, subsequent calls return
 * the same sample data object.
 */
IPSampleData *
instp_sample_data_get_blank (void)
{
  static IPSampleData *blank_sampledata = NULL;
  IPSampleStore *store;

  if (!blank_sampledata) /* blank sampledata already created? */
    {
      void *data;

      /* allocate blank minimum sample size */
      if (!(data = g_malloc0 (48 * 2)))
	return (NULL);

      /* create the new sample data object */
      if (!(blank_sampledata = instp_sample_data_new ()))
	{
	  g_free (data);
	  return (NULL);
	}

      /* set size and create a RAM sample store */
      instp_sample_data_set_size (blank_sampledata, 48);
      store = instp_sample_store_new (blank_sampledata, IPSAMPLE_METHOD_RAM);
      if (!store)
	{
	  g_free (data);
	  return (NULL);
	}

      instp_sample_method_RAM_set_pointer (blank_sampledata, store, data);

      /* permanent reference to the blank sample data, it will always exist */
      instp_item_ref (INSTP_ITEM (blank_sampledata));
    }

  return (blank_sampledata);
}


/* -------------------------- */
/* ---- IPZone functions ---- */
/* -------------------------- */

/* #IPItem_DoneFunc for #IPZone items */
void
instp_zone_done_func (IPItem *item)
{
  IPZone *zone;
  IPGen *gen;
  IPMod *mod;
  void *temp;

  g_return_if_fail (item != NULL);
  g_return_if_fail (INSTP_IS_ZONE (item));

  zone = INSTP_ZONE (item);

  gen = instp_zone_first_gen (zone);
  while (gen)
    {				/* Free IPGens for this zone */
      temp = gen;
      gen = instp_gen_next (gen);
      instp_gen_free (temp);
    }
  zone->gen = NULL;

  mod = instp_zone_first_mod (zone);
  while (mod)
    {				/* Free IPMods for this zone */
      temp = mod;
      mod = instp_mod_next (mod);
      instp_mod_free (temp);
    }
  zone->mod = NULL;
}

/* #IPItem_DuplicateFunc for #IPZone items */
IPItem *
instp_zone_duplicate_func (const IPItem *item)
{
  IPZone *zone, *dup;
  IPGen *gen, *dupgen;
  IPMod *mod, *dupmod;

  g_return_val_if_fail (item != NULL, NULL);
  g_return_val_if_fail (INSTP_IS_ZONE (item), NULL);

  zone = INSTP_ZONE (item);

  dup = instp_zone_new ();
  if (!dup) return (NULL);

  dup->sfitem.userptr = zone->sfitem.userptr;

  instp_zone_set_refitem (dup, zone->refitem);

  gen = instp_zone_first_gen (zone);
  while (gen)			/* loop over zone IPGens */
    {
      dupgen = instp_gen_new ();
      if (!dupgen)
	{
	  instp_item_destroy (INSTP_ITEM (dup));
	  return (NULL);
	}

      dupgen->id = gen->id;
      dupgen->amount = gen->amount;

      instp_zone_insert_gen (dup, dupgen, -1);
      gen = instp_gen_next (gen);
    }

  mod = instp_zone_first_mod (zone);
  while (mod)			/* loop over zone IPMods */
    {
      dupmod = instp_mod_new ();
      if (!dupmod)
	{
	  instp_item_destroy (INSTP_ITEM (dup));
	  return (NULL);
	}

      dupmod->src = mod->src;
      dupmod->dest = mod->dest;
      dupmod->amount = mod->amount;
      dupmod->amtsrc = mod->amtsrc;
      dupmod->trans = mod->trans;

      instp_zone_insert_mod (dup, dupmod, -1);
      mod = instp_mod_next (mod);
    }

  return (INSTP_ITEM (dup));
}

/**
 * Get a zones first generator
 * @zone Zone to get generator from
 * Returns: First Generator in Zone's generator list or NULL if none
 */
IPGen *
instp_zone_first_gen (const IPZone *zone)
{
  g_return_val_if_fail (zone != NULL, NULL);

  return (zone->gen);
}

/**
 * Get a zones first modulator
 * @zone Zone to get modulator from
 * Returns: First Modulator in Zone's modulator list or NULL if none
 */
IPMod *
instp_zone_first_mod (const IPZone *zone)
{
  g_return_val_if_fail (zone != NULL, NULL);
  return (zone->mod);
}

/**
 * Set a zone's referenced item pointer
 * @zone Zone to set reference pointer of
 * @refitem Pointer to referenced item or NULL for global zone
 *
 * Set the item reference of a zone. Unreferences previously referenced
 * item (if any) and references new one (if not NULL).
 * Sets changed flag.
 */
void
instp_zone_set_refitem (IPZone *zone, IPItem *refitem)
{
  g_return_if_fail (zone != NULL);

  if (refitem) instp_item_ref (refitem);
  if (zone->refitem) instp_item_unref (zone->refitem);

  zone->refitem = (IPItem *)refitem;
  instp_item_set_root_changed_flag (INSTP_ITEM (zone), TRUE);
}

/**
 * Check if a key and velocity falls in a zone's ranges
 * @zone Zone to check if in range
 * @note MIDI note number or -1 for wildcard
 * @velocity MIDI velocity or -1 for wildcard
 * Returns: TRUE if zone is in key and velocity range, FALSE otherwise
 */
gboolean
instp_zone_in_range (IPZone *zone, int note, int velocity)
{
  IPGen *gen;
  IPGenAmount *keyamt = NULL, *velamt = NULL;
  gboolean in_range;

  g_return_val_if_fail (zone != NULL, FALSE);

  gen = zone->gen;
  if (gen && gen->id == IPGEN_KEY_RANGE)
    {
      if (note != -1)
	keyamt = &gen->amount;
      gen = gen->next;
    }
  if (gen && gen->id == IPGEN_VELOCITY_RANGE)
    if (velocity != -1)
      velamt = &gen->amount;

  in_range = (!keyamt || (note >= keyamt->range.low
			  && note <= keyamt->range.high))
    && (!velamt || (velocity >= velamt->range.low
		    && velocity <= velamt->range.high));

  return (in_range);
}

/**
 * Insert a generator into a zone (careful!)
 * @zone Zone to insert generator into
 * @gen Generator to insert
 * @pos Index in zone's generator list to insert into
 * (0 = first, < 0 = last)
 * \see instp_zone_set_gen
 *
 * Inserts a generator into a zone's generator list at the given position.
 * Does \b NOT check for duplicates or ensure ordering! See
 * #instp_zone_set_gen for the usual way to set a generator.
 * Sets changed flag.
 */
void
instp_zone_insert_gen (IPZone *zone, IPGen *gen, int pos)
{
  IPGen *p, *prev;

  g_return_if_fail (zone != NULL);
  g_return_if_fail (gen != NULL);

  instp_item_set_root_changed_flag (INSTP_ITEM (zone), TRUE);

  G_LOCK (instp_voice_lock);

  p = instp_zone_first_gen (zone);
  prev = NULL;

  if (pos == 0 || !p)
    {
      gen->next = p;
      zone->gen = gen;

      G_UNLOCK (instp_voice_lock);

      return;
    }

  if (pos < 0)
    while (p)
      {
	prev = p;
	p = instp_gen_next (p);
      }
  else
    while (p && pos--)
      {
	prev = p;
	p = instp_gen_next (p);
      }

  gen->next = prev->next;
  prev->next = gen;

  G_UNLOCK (instp_voice_lock);
}

/**
 * Get a generator value from a zone
 * @zone Zone to get generator value from
 * @genid Generator ID (#IPGenType) of value to get
 * @out_amt Output: Pointer to store generator amount to
 * Returns: Boolean value: TRUE if generator is set in Zone or the generators
 *   are stored as an array; FALSE if not set, in which case the value
 *  stored to output_amt is the default value for the given generator ID.
 */
gboolean
instp_zone_get_gen (const IPZone *zone, guint16 genid,
		    IPGenAmount *out_amt)
{
  IPGen *gen;
  IPGenAmount defamt;
  IPItem *parent;
  gboolean ispreset = FALSE;

  g_return_val_if_fail (zone != NULL, FALSE);
  g_return_val_if_fail (out_amt != NULL, FALSE);

  gen = instp_zone_first_gen (zone);
  while (gen)    		/* loop through zone's generators */
    {
      if (gen->id == genid)	/* Is this gen the requested one? */
	{
	  *out_amt = gen->amount;
	  return (TRUE);	/* generator set in zone */
	}
      gen = instp_gen_next (gen);
    }

  parent = instp_item_parent (INSTP_ITEM (zone));
  if (parent && INSTP_IS_PRESET (parent))
    ispreset = TRUE;

  defamt = instp_genid_default_value (genid, ispreset);
  *out_amt = defamt;		/* default amount for this generator */
  return (FALSE);		/* generator not set in zone */
}

/**
 * Set a generator value in a zone
 * @zone Zone to set generator in
 * @genid Generator ID (#IPGenType) of generator to set
 * @amount Value to set generator to
 * Returns: INSTP_OK on success or INSTP_FAIL otherwise
 * 
 * Set a generator value in a zone. This is the common "safe" way to set a
 * generator. It overwrites existing duplicate generators and keeps the
 * generator list in correct order.
 * Sets changed flag.
 */
int
instp_zone_set_gen (IPZone *zone, guint16 genid, IPGenAmount amt)
{
  IPGen *gen, *lastgen = NULL, *newgen;
  int pos;

  g_return_val_if_fail (zone != NULL, INSTP_FAIL);

  G_LOCK (instp_voice_lock);

  gen = instp_zone_first_gen (zone);
  while (gen)			/* loop through zone's generators */
    {
      if (gen->id == genid)
	{			/* Is this gen the requested one? */
	  if (gen->amount.sword != amt.sword) /* has same value? */
	    {
	      gen->amount.sword = amt.sword;
	      instp_item_set_root_changed_flag (INSTP_ITEM (zone), TRUE);
	    }
	  G_UNLOCK (instp_voice_lock);
	  return (INSTP_OK);
	}
      lastgen = gen;
      gen = instp_gen_next (gen);
    }

  G_UNLOCK (instp_voice_lock);

  /* Generator not found, and amt isn't the default */
  newgen = instp_gen_new ();
  if (!newgen) return (INSTP_FAIL);

  newgen->id = genid;
  newgen->amount.sword = amt.sword;

  pos = -1;			/* default: index -1 appends generators */

  /* Key ranges must be first */
  if (genid == IPGEN_KEY_RANGE) pos = 0;
  else if (genid == IPGEN_VELOCITY_RANGE)
    {				/* Velranges preceded only by keyranges */
      pos = 0;
      gen = instp_zone_first_gen (zone);
      if (gen && gen->id == IPGEN_KEY_RANGE) pos++;
    }

  instp_zone_insert_gen (zone, newgen, pos);
  instp_item_set_root_changed_flag (INSTP_ITEM (zone), TRUE);

  return (INSTP_OK);
}

/**
 * Unset a generator in a zone
 * @zone Zone to unset generator in
 * @genid Generator ID (#IPGenType) to unset
 *
 * Unsets a generator in a zone by removing it from the zone's generator list,
 * thereby using its default value.
 * Sets changed flag.
 */
void
instp_zone_unset_gen (IPZone *zone, guint16 genid)
{
  IPGen *gen, *prevgen = NULL;

  g_return_if_fail (zone != NULL);

  G_LOCK (instp_voice_lock);

  gen = instp_zone_first_gen (zone);
  while (gen)			/* loop over generators */
    {
      if (gen->id == genid)
	{
	  if (prevgen) prevgen->next = gen->next; /* remove from list */
	  else zone->gen = gen->next; /* update root gen ptr */

	  instp_gen_free (gen);

	  G_UNLOCK (instp_voice_lock);
	  instp_item_set_root_changed_flag (INSTP_ITEM (zone), TRUE);
	  return;
	}
      prevgen = gen;
      gen = instp_gen_next (gen);
    }

  G_UNLOCK (instp_voice_lock);
}

/**
 * Count number of generators in a zone
 * @zone Zone to count generators in
 * Returns: Count of generators
 */
int
instp_zone_gen_count (IPZone *zone)
{
  IPGen *gen;
  int i = 0;

  gen = zone->gen;
  while (gen)
    {
      i++;
      gen = gen->next;
    }

  return (i);
}

/**
 * Insert a modulator into a zone (careful!)
 * @zone Zone to insert into
 * @mod Modulator to insert
 * @pos Index position in zone's modulator list to insert into
 * (0 = first, < 0 = last)
 *
 * Inserts a modulator into a zones modulator list. Does not check for
 * duplicates!
 * Sets changed flag.
 */
void
instp_zone_insert_mod (IPZone *zone, IPMod *mod, int pos)
{
  g_return_if_fail (zone != NULL);
  g_return_if_fail (mod != NULL);

  G_LOCK (instp_voice_lock);
  zone->mod = instp_mod_list_insert (zone->mod, mod, pos);
  instp_item_set_root_changed_flag (INSTP_ITEM (zone), TRUE);
  G_UNLOCK (instp_voice_lock);
}

/**
 * Remove a modulator from a zone by pointer
 * @zone Zone to remove modulator from
 * @mod Pointer of modulator to remove
 */
void
instp_zone_remove_mod_by_ptr (IPZone *zone, IPMod *mod)
{
  g_return_if_fail (zone != NULL);
  g_return_if_fail (mod != NULL);

  G_LOCK (instp_voice_lock);
  zone->mod = instp_mod_list_remove_by_ptr (zone->mod, mod);
  G_UNLOCK (instp_voice_lock);
}

/**
 * Count number of modulators in a zone
 * @zone Zone to count modulators in
 * Returns: Count of modulators
 */
int
instp_zone_mod_count (IPZone *zone)
{
  IPMod *mod;
  int i = 0;

  mod = zone->mod;
  while (mod)
    {
      i++;
      mod = mod->next;
    }

  return (i);
}

/* ------------------------- */
/* ---- IPGen functions ---- */
/* ------------------------- */

/**
 * Get next generator in list
 * @gen Generator to get next from
 * Returns: The next generator in the generator list or NULL if no more
 */
IPGen *
instp_gen_next (const IPGen *gen)
{
  g_return_val_if_fail (gen != NULL, NULL);
  return (gen->next);
}

/**
 * Create a new generator object
 * Returns: New generator or NULL on error
 */
IPGen *
instp_gen_new (void)
{
  return (g_malloc0 (sizeof (IPGen)));
}

/**
 * Free an IPGen structure
 * @gen IPGen to free
 *
 * Frees an IPGen structure, does \b NOT disconnect from parent zone (if any).
 */
void
instp_gen_free (IPGen *gen)
{
  g_return_if_fail (gen != NULL);
  g_free (gen);
}

/**
 * Check validity of generator ID
 * @genid Generator ID (see #IPGenType)
 * @ispreset TRUE if its a Preset generator, FALSE if Instrument
 * Returns: TRUE if generator is valid for the specified object
 *   (Preset/Instrument)
 */
gboolean
instp_genid_is_valid (int genid, gboolean ispreset)
{				/* is generator id valid? */
  guint16 badgen[] = {
    /* bad generators for instruments or presets */

    IPGEN_UNUSED1, IPGEN_UNUSED2, IPGEN_UNUSED3, IPGEN_UNUSED4,
    IPGEN_RESERVED1, IPGEN_RESERVED2, IPGEN_RESERVED3,

    /* bad preset generators */

    IPGEN_START_ADDR_OFS, IPGEN_END_ADDR_OFS, IPGEN_START_LOOP_ADDR_OFS,
    IPGEN_END_LOOP_ADDR_OFS, IPGEN_START_ADDR_COARSE_OFS,
    IPGEN_END_ADDR_COARSE_OFS, IPGEN_START_LOOP_ADDR_COARSE_OFS, IPGEN_KEY_NUM,
    IPGEN_VELOCITY, IPGEN_END_LOOP_ADDR_COARSE_OFS, IPGEN_SAMPLE_MODES,
    IPGEN_EXCLUSIVE_CLASS, IPGEN_OVERRIDE_ROOT_KEY
  };

  int i = ispreset ? sizeof (badgen) / sizeof (guint16) : 7;

  if (genid < 0 || genid >= IPGEN_COUNT) return (FALSE);

  while (i--)
    if (genid == badgen[i]) return (FALSE);

  return (TRUE);
}

/**
 * Get default value for a generator ID
 * @genid Generator ID
 * @ispreset TRUE for preset generators, FALSE for instrument
 */
IPGenAmount
instp_genid_default_value (guint16 genid, gboolean ispreset)
{
  IPGenAmount amt;

  amt.sword = 0;
  g_return_val_if_fail (instp_genid_is_valid (genid, ispreset), amt);

  if (ispreset)
    {
      if (instp_gen_info[genid].unit == IPUNIT_RANGE)
	{
	  amt.range.low = 0;
	  amt.range.high = 127;
	  return (amt);
	}
      else return (amt);
    }
  else
    {
      amt.sword = instp_gen_info[genid].def;
      return (amt);
    }
}

/**
 * Offset a generator amount
 * @genid ID of Generator to offset. Must be a valid preset generator.
 * @abs Pointer to the initial amount to offset, result is stored back
 *   into this parameter.
 * @ofs Offset amount.
 * Returns: TRUE if value was clamped, FALSE otherwise.
 *
 * Offsets a generator amount. Result of offset is clamped to maximum and
 *   minimum values for the given generator ID.
 */
gboolean
instp_genid_offset (int genid, IPGenAmount *dst, IPGenAmount ofs)
{
  gint32 temp;
  gboolean clamped = FALSE;

  g_return_val_if_fail (dst != NULL, FALSE);
  g_return_val_if_fail (instp_genid_is_valid (genid, TRUE), FALSE);

  if (genid != IPGEN_KEY_RANGE && genid != IPGEN_VELOCITY_RANGE)
    {
      temp = (gint32) (dst->sword) + (gint32) (ofs.sword);
      if (temp < (gint32) instp_gen_info[genid].min)
	{
	  temp = instp_gen_info[genid].min;
	  clamped = TRUE;
	}
      else if (temp > (gint32) instp_gen_info[genid].max)
	{
	  temp = instp_gen_info[genid].max;
	  clamped = TRUE;
	}
      dst->sword = (gint16) temp;
    }
  else *dst = instp_units_range_intersect (*dst, ofs);

  return (clamped);
}

/**
 * Allocate space for generator amount array
 * Returns: Pointer to generator amount array or NULL on error
 * (\b free when no longer needed)
 *
 * Allocates space for an array of generator amounts, with enough space to
 * store all known generator values
 */
IPGenAmount *
instp_gen_array_alloc (void)
{
  return (g_malloc (sizeof (IPGenAmount) * IPGEN_COUNT));
}

/**
 * Initialize a generator amount array
 * @genarr Generator amount array returned by #instp_genarr_alloc
 * @offset TRUE = initialize to Preset offset (zero) values,
 * FALSE = initialize to instrument default values
 */
void
instp_gen_array_init (IPGenAmount *genarr, gboolean offset)
{
  int i;

  g_return_if_fail (genarr != NULL);

  if (offset)			/* set to offset generator default values */
    {
      for (i = 0; i < IPGEN_COUNT; i++)
	genarr[i].uword = 0;

      genarr[IPGEN_KEY_RANGE].uword = DEFRANGE;
      genarr[IPGEN_VELOCITY_RANGE].uword = DEFRANGE;
    }
  else				/* set to absolute generator default values */
    {
      for (i = 0; i < IPGEN_COUNT; i++) /* for each generator item */
	genarr[i].uword = instp_gen_info[i].def; /* init to default val */
    }
}

/**
 * Copy values from one generator array to another
 * @dest Destination generator array
 * @src Source generator array
 */
void
instp_gen_array_copy (IPGenAmount *dest, const IPGenAmount *src)
{
  memcpy (dest, src, sizeof (IPGenAmount) * IPGEN_COUNT);
}

/**
 * Process a zone's generators into a generator amount array
 * @genarr Destination generator amount array where values will be stored
 * @zone Zone to process (extract) generator values from
 */
void
instp_gen_array_process_zone (IPGenAmount *genarr, const IPZone *zone)
{
  IPGen *gen;

  g_return_if_fail (genarr != NULL);
  g_return_if_fail (zone != NULL);

  gen = instp_zone_first_gen (zone); /* first generator */
  while (gen)			/* loop over generators */
    {
      genarr[gen->id] = gen->amount; /* assign to array */
      gen = instp_gen_next (gen);
    }
}

/**
 * Offset an array of absolute generator values with offset values
 * @abs Destination generator amount array that contains absolute
 * (Instrument) generator values
 * @ofs Source generator amount array that contains offset (Preset)
 * generator values
 *
 * Offsets the generators amounts in abs by adding to it the values in
 * ofs. Values are clamped to their valid ranges.
 */
void
instp_gen_array_offset (IPGenAmount *abs, const IPGenAmount *ofs)
{
  int i;
  gint32 temp;

  for (i = 0; i < IPGEN_COUNT; i++)
    {
      /* skip over KEY_RANGE and VELOCITY_RANGE */
      if (i == IPGEN_KEY_RANGE) i += 2;

      temp = (gint32) (abs[i].sword) + (gint32) (ofs[i].sword);
      if (temp < (gint32) instp_gen_info[i].min)
	temp = instp_gen_info[i].min;
      else if (temp > (gint32) instp_gen_info[i].max)
	temp = instp_gen_info[i].max;
      abs[i].sword = (gint16) temp;
    }

  abs[IPGEN_KEY_RANGE] =
    instp_units_range_intersect (abs[IPGEN_KEY_RANGE],
				 ofs[IPGEN_KEY_RANGE]);
  abs[IPGEN_VELOCITY_RANGE] =
    instp_units_range_intersect (abs[IPGEN_VELOCITY_RANGE],
				 ofs[IPGEN_VELOCITY_RANGE]);
}

/* ------------------------- */
/* ---- IPMod functions ---- */
/* ------------------------- */

/**
 * Get next modulator in list
 * @mod Modulator to get next from
 * Returns: The next modulator in the modulator list or NULL if no more
 */
IPMod *
instp_mod_next (const IPMod *mod)
{
  g_return_val_if_fail (mod != NULL, NULL);
  return (mod->next);
}

/**
 * Create a new modulator object
 * Returns: New modulator or NULL on error
 */
IPMod *
instp_mod_new (void)
{
  return (g_malloc0 (sizeof (IPMod)));
}

/**
 * Duplicate a modulator
 * @mod Modulator to duplicate
 * Returns: New duplicate modulator or NULL on error
 */
IPMod *
instp_mod_duplicate (IPMod *mod)
{
  IPMod *newmod;

  g_return_val_if_fail (mod != NULL, NULL);

  newmod = instp_mod_new ();
  if (!newmod) return (NULL);

  newmod->src = mod->src;
  newmod->dest = mod->dest;
  newmod->amount = mod->amount;
  newmod->amtsrc = mod->amtsrc;
  newmod->trans = mod->trans;

  return (newmod);
}

/**
 * Free an IPMod structure
 * @mod IPMod to free, does \b NOT disconnect from parent zone (if any).
 */
void
instp_mod_free (IPMod *mod)
{
  g_return_if_fail (mod != NULL);
  g_free (mod);
}

/**
 * Insert a modulator into a modulator list
 * @list Root modulator of list (or NULL if empty list)
 * @mod Modulator to insert into list
 * @pos Index in list to insert modulator into (0 = first, < 0 = last)
 * Returns: New root node of list
 */
IPMod *
instp_mod_list_insert (IPMod *list, IPMod *mod, int pos)
{
  IPMod *p;
  int i = 0;

  g_return_val_if_fail (mod != NULL, list);

  if (pos == 0 || !list)	/* insert before root modulator? */
    {
      mod->next = list;
      return (mod);
    }

  if (pos == -1) pos = 0x7FFFFFFF;

  p = list;
  while (p->next && ++i < pos)
    p = p->next;

  mod->next = p->next;
  p->next = mod;

  return (list);
}

/**
 * Remove a modulator from a list by pointer
 * @list Modulator list root
 * @mod Pointer of modulator to remove
 * @New root of modulator list
 */
IPMod *
instp_mod_list_remove_by_ptr (IPMod *list, IPMod *mod)
{
  IPMod *p, *prev = NULL;

  g_return_val_if_fail (mod != NULL, list);

  p = list;
  while (p)			/* loop over modulators in list */
    {
      if (p == mod)		/* the modulator to delete? */
	{
	  if (prev) prev->next = p->next;
	  p = instp_mod_next (p);
	  instp_mod_free (mod);
	  return (prev ? list : p);
	}
      prev = p;
      p = instp_mod_next (p);
    }

  return (list);
}

/**
 * Copy a modulator list
 * @list First modulator of list to copy (empty list is allowed)
 * Returns: First modulator of new copied list or NULL on error or empty list
 */
IPMod *
instp_mod_list_copy (IPMod *list)
{
  IPMod *newlist = NULL;
  IPMod *dup;

  while (list)
    {
      dup = instp_mod_duplicate (list);
      if (!dup)
	{
	  instp_mod_list_free (newlist);
	  return (NULL);
	}
      newlist = instp_mod_list_insert (newlist, dup, -1); /* append */
      list = instp_mod_next (list);
    }

  return (newlist);
}

/**
 * Combine two modulator lists and override duplicates
 * @amods List of modulators, duplicates are replaced by 'bmods'.
 * @bmods Second list of modulators, modulators in this list that are
 *   duplicates override the modulators in 'amods'.
 * Returns: New list of modulators created by combining the two lists.
 *   Modulators in the second list override ones in the first for duplicates.
 */
IPMod *
instp_mod_list_combine (IPMod *amods, IPMod *bmods)
{
  IPMod *newmods = NULL;
  IPMod *newamods;
  IPMod *p;

  if (amods)			/* copy amods list */
    if (!(newmods = instp_mod_list_copy (amods)))
      return (NULL);

  newamods = newmods;

  while (bmods)			/* loop over bmods */
    {
      p = newamods;
      while (p)
	{
	  if (IPMOD_ARE_IDENTICAL (p, bmods)) break;
	  p = instp_mod_next (p);
	}

      if (!p)			/* no duplicates, just add it to list */
	{
	  p = instp_mod_duplicate (bmods);
	  if (!p)
	    {
	      instp_mod_list_free (newmods);
	      return (NULL);
	    }
	  newmods = instp_mod_list_insert (newmods, p, 0); /* prepend */
	}
      else p->amount = bmods->amount; /* override duplicate */

      bmods = instp_mod_next (bmods);
    }

  return (newmods);
}

/**
 * Combine two modulator lists and add duplicates
 * @amods First modulator list
 * @bmods Second modulator list
 * Returns: First modulator of new list made by combining both lists and
 *   adding duplicates (modulator amount field), or NULL on error.
 */
IPMod *
instp_mod_list_offset (IPMod *amods, IPMod *bmods)
{
  IPMod *newmods = NULL, *new_amods;
  IPMod *p;
  int add;

  if (amods)			/* copy amods list */
    if (!(newmods = instp_mod_list_copy (amods)))
      return (NULL);

  new_amods = newmods;

  while (bmods)			/* loop over bmods */
    {
      p = new_amods;
      while (p)
	{
	  if (IPMOD_ARE_IDENTICAL (p, bmods)) break;
	  p = instp_mod_next (p);
	}

      if (!p)			/* no duplicates, just add it to list */
	{
	  p = instp_mod_duplicate (bmods);
	  if (!p)
	    {
	      instp_mod_list_free (newmods);
	      return (NULL);
	    }
	  newmods = instp_mod_list_insert (newmods, p, 0); /* prepend */
	}
      else			/* add duplicate amounts */
	{
	  add = p->amount + bmods->amount;
	  if (add > 32767) add = 32767;
	  else if (add < -32768) add = -32768;
	  p->amount = add;
	}

      bmods = instp_mod_next (bmods);
    }

  return (newmods);
}

/**
 * Free a list of modulators
 * @mod Root modulator in list to free
 */
void
instp_mod_list_free (IPMod *mod)
{
  IPMod *p, *temp;

  p = mod;
  while (p)
    {
      temp = p;
      p = instp_mod_next (p);
      instp_mod_free (temp);
    }
}

/**
 * Returns a list of default instrument modulators
 * Returns: First modulator in list of default modulators, should \b NOT be
 *   modified or freed.
 */
IPMod *
instp_mod_list_default (void)
{
  IPMod default_mods[] =
  {
    { 0x0502, IPGEN_ATTENUATION, 960, 0x0, 0 },
    { 0x0102, IPGEN_FILTER_FC, -2400, 0xD02, 0 },
    { 0x000D, IPGEN_VIB_LFO_TO_PITCH, 50, 0x0, 0 },
    { 0x0081, IPGEN_VIB_LFO_TO_PITCH, 50, 0x0, 0 },
    { 0x0587, IPGEN_ATTENUATION, 960, 0x0, 0 },
    { 0x028A, IPGEN_PAN, 1000, 0x0, 0 },
    { 0x058B, IPGEN_ATTENUATION, 960, 0x0, 0 },
    { 0x00DB, IPGEN_REVERB_SEND, 200, 0x0, 0 },
    { 0x00DD, IPGEN_CHORUS_SEND, 200, 0x0, 0 }
    //    { 0x020E, InitialPitch WTF?, 12700, 0x0010, 0 },
  };
  static IPMod *mod_list = NULL;
  IPMod *mod;
  int i;

  if (!mod_list)
    for (i = 0; i < sizeof (default_mods) / sizeof (IPMod); i++)
      {
	mod = instp_mod_new ();
	if (!mod)
	  {
	    instp_mod_list_free (mod_list);
	    mod_list = NULL;
	    return (NULL);
	  }

	*mod = default_mods[i];
	mod_list = instp_mod_list_insert (mod_list, mod, -1);
      }

  return (mod_list);
}

/* dump a list of modulators to stdout for debugging */
void
instp_mod_list_dump (IPMod *mod)
{
  while (mod)
    {
      printf ("-MOD src=%x amtsrc=%x dest=%d amount=%d\n", mod->src,
	      mod->amtsrc, mod->dest, mod->amount);
      mod = instp_mod_next (mod);
    }
}

/* unit conversion functions */

/**
 * Convert a sound font generator amount to user units
 * @genid Generator ID
 * @amt Generator amount
 * @out_userval Output: Pointer to return user units value in
 * @ispreset TRUE if its a Preset generator, FALSE if Instrument
 */
void
instp_units_sfont_to_user (guint16 genid, IPGenAmount amt,
			   float *out_userval, gboolean ispreset)
{
  int unit;
  float val;

  g_return_if_fail (instp_genid_is_valid (genid, ispreset));

  unit = instp_gen_info[genid].unit; /* get the unit enum for this gen */
  if (unit == IPUNIT_NONE || unit == IPUNIT_RANGE)
    {
      g_critical (IPERR_MSG_UNKNOWN_UNIT_TYPE_1, unit);
      return;
    }

  /* convert absolute (instrument) or offset (preset) gen value? */
  if (ispreset)
    val = (*instp_gen_conv[unit].sf2ofs) (amt); /* convert ofs gen */
  else
    val = (*instp_gen_conv[unit].sf2user) (amt); /* convert abs gen */

  /* call us with no val ptr and u shall have no info */
  if (out_userval) *out_userval = val;
}

/**
 * Convert a sound font generator amount to user units and formatted
 *   string
 * @genid Generator ID
 * @amt Generator amount
 * @out_userval Output: Pointer to return user units value in
 *   (or NULL to ignore)
 * @out_userstr Output: Pointer to return user units formatted string in
 *   (or NULL to ignore). \b Free userstr when finished with it.
 *   Could be NULL if error occurs (no memory, invalid genid).
 * @ispreset TRUE if its a Preset generator, FALSE if Instrument
 */
void
instp_units_sfont_to_user_str (guint16 genid, IPGenAmount amt,
			       float *out_userval, char **out_userstr,
			       gboolean ispreset)
{
  float val;
  int unit;
  char *s;

  /* initialize to default values in case of error */
  if (out_userval) *out_userval = 0.0;
  if (out_userstr) *out_userstr = NULL;

  instp_units_sfont_to_user (genid, amt, &val, ispreset);

  unit = instp_gen_info[genid].unit;
  s = g_strdup_printf ("%.*f", instp_gen_conv[unit].digits, val);

  if (out_userval) *out_userval = val;
  if (out_userstr) *out_userstr = s;
}

/**
 * Convert user units to sound font generator units
 * @genid Generator ID (#IPGenType)
 * @userval User units value
 * @ispreset TRUE if its a Preset generator, FALSE if Instrument
 * @clamp TRUE to clamp sound font units result to generators valid
 * range
 * Returns: Value converted to sound font units
 */
int
instp_units_user_to_sfont (guint16 genid, float userval, gboolean ispreset,
			   gboolean clamp)
{
  int unit;
  int val;

  g_return_val_if_fail (instp_genid_is_valid (genid, ispreset), 0);

  unit = instp_gen_info[genid].unit;
  if (unit == IPUNIT_NONE || unit == IPUNIT_RANGE)
    {
      g_critical (IPERR_MSG_UNKNOWN_UNIT_TYPE_1, unit);
      return (0);
    }

  /* calculate value in sound font units */
  if (ispreset) val = (*instp_gen_conv[unit].ofs2sf) (userval);
  else val = (*instp_gen_conv[unit].user2sf) (userval);

  /* clamp value if requested */
  if (clamp) instp_units_clamp (genid, &val, ispreset);

  return (val);
}

/**
 * Clamp a generators value to its valid range
 * @genid Generator ID (#IPGenType)
 * @sfval Generator value to clamp (changed in place)
 * @ispreset TRUE if its a Preset generator, FALSE if Instrument
 */
void
instp_units_clamp (guint16 genid, int *sfval, gboolean ispreset)
{
  int ofsrange;			/* used only for offset gens, range of value */

  g_return_if_fail (instp_genid_is_valid (genid, ispreset));

  if (ispreset)
    {
      ofsrange = instp_gen_info[genid].max - instp_gen_info[genid].min;
      if (*sfval < -ofsrange) *sfval = -ofsrange;
      else if (*sfval > ofsrange) *sfval = ofsrange;
    }
  else
    {
      if (*sfval < instp_gen_info[genid].min)
	*sfval = instp_gen_info[genid].min;
      else if (*sfval > instp_gen_info[genid].max)
	*sfval = instp_gen_info[genid].max;
    }
}

/**
 * Find intersection of two generator ranges
 * @a First generator amount range
 * @b Second generator amount range
 * Returns: Pointer to generator amount produced by intersecting the two input
 *   ranges (common shared range). If ranges don't share anything in common
 *   returns a 0-0 range (note that this is technically a valid 1 note range).
 */
IPGenAmount
instp_units_range_intersect (const IPGenAmount a, const IPGenAmount b)
{
  int i = 0;
  guint8 p[2] = { 0, 0 };
  IPGenAmount val;

  if ((a.range.low >= b.range.low) && (a.range.low <= b.range.high))
    p[i++] = a.range.low;

  if ((a.range.high >= b.range.low) && (a.range.high <= b.range.high))
    p[i++] = a.range.high;

  if ((b.range.low > a.range.low) && (b.range.low < a.range.high))
    p[i++] = b.range.low;

  if ((b.range.high > a.range.low) && (b.range.high < a.range.high))
    p[i++] = b.range.high;

  if (p[0] < p[1])
    {
      val.range.low = p[0];
      val.range.high = p[1];
    }
  else
    {
      val.range.low = p[1];
      val.range.high = p[0];
    }

  return (val);
}

/**
 * Unit conversion function: Float to integer
 * @f Floating number
 * Returns: Converted integer
 */
int
instp_float2int (float f)
{
  return ((int) f);
}

/**
 * Unit conversion function: Integer to float
 * @val Generator amount
 * Returns: Converted floating point value
 */
float
instp_int2float (IPGenAmount val)
{
  return ((float) val.sword);
}

/**
 * Unit conversion function: Hertz to cents
 * @hz Floating point Hertz (cycles per second) value
 * Returns: Converted value in cents (100ths of a Semitone (12th of an octave))
 */
int
instp_hz2cents (float hz)
{
  return ((int) (log (hz / 8.176) / log (2) * 1200));
}

/**
 * Unit conversion function: Cents to Hertz
 * @cents Generator amount in Cents
 * Returns: Converted floating point value in Hertz (cycles per second)
 */
float
instp_cents2hz (IPGenAmount cents)
{
  return (8.176 * pow (2.0, ((double) cents.sword) / 1200.0));
}

/**
 * Unit conversion function: Decibels to Centibels (10ths of a dB)
 * @db Floating point value in Decibels
 * Returns: Converted integer in Centibels (10ths of a Decibel)
 */
int
instp_db2cb (float db)
{
  return ((int) (db * 10.0));
}

/**
 * Unit conversion function: Centibels to Decibels
 * @cb Generator amount in Centibels
 * Returns: Converted floating point value in Decibels
 */
float
instp_cb2db (IPGenAmount cb)
{
  return ((float) cb.sword / 10.0);
}

/**
 * Unit conversion function: Seconds to timecents (1200th root of 2)
 * @sec Floating point value in seconds
 * Returns: Converted integer in timecents (1200th root of 2)
 */
int
instp_sec2tcent (float sec)
{
  return ((int) (log (sec) / log (2) * 1200));
}

/**
 * Unit conversion function: Timecents to seconds
 * @tcent Generator amount in timecents
 * Returns: Converted float point value in seconds
 */
float
instp_tcent2sec (IPGenAmount tcent)
{
  return (pow (2.0, ((double) tcent.sword) / 1200.0));
}

/**
 * Unit conversion function: Percent to tenths percent
 * @perc Floating point value in percent
 * Returns: Converted integer in 10ths of a percent
 */
int
instp_perc2tperc (float perc)
{
  return ((int) (perc * 10));
}

/**
 * Unit conversion function: Tenths percent to percent
 * @tperc Generator amount in tenths of a percent
 * Returns: Converted floating point value in percent
 */
float
instp_tperc2perc (IPGenAmount tperc)
{
  return (((float) tperc.sword) / 10);
}
