#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

// Denormalise floats, only actually needed for PIII and very recent PowerPC
#define DENORMALISE(fv) (((*(unsigned int*)&(fv))&0x7f800000)==0)?0.0f:(fv)

// Limit value between given ranges
#define LIMIT(v,l,u) (v<l?l:(v>u?u:v))

// A fast, truncating towards 0 modulo function. ANSI C doesn't define
// which % will do, most truncate towards -inf
#define MOD(v,m) (v<0?v+m:(v>m?v-m:v))

#define F_R 3
#define F_Q 0.001

// Oscilator instability
#define INST 0.2

LADSPA_Data *sin_tbl, *tri_tbl, *saw_tbl, *squ_tbl;
int tbl_ref_count = 0;
long sample_rate;

inline float *wave_tbl(float wave) {
        if (wave < 1.0f) {
                return sin_tbl;
        } else if (wave < 2.0f) {
                return tri_tbl;
        } else if (wave < 3.0f) {
                return saw_tbl;
        } else if (wave < 4.0f) {
                return squ_tbl;
        }
        return sin_tbl;
}

/* Structure to hold parameters for SV filter */

typedef struct {
        float f;     // 2.0*sin(PI*fs/(fc*r));
        float q;     // 2.0*cos(pow(q, 0.1)*PI*0.5);
        float qnrm;  // sqrt(m/2.0f+0.01f);
        float h;     // high pass output
        float b;     // band pass output
        float l;     // low pass output
} sv_filter;

/* Store data in SVF struct, takes the sampling frequency, cutoff frequency
   and Q, and fills in the structure passed */

inline void setup_svf(sv_filter *sv, float fs, float fc, float q) {
        sv->f = 2.0f * sin(M_PI * fc / (float)(fs * F_R));
        sv->q = 2.0f * cos(pow(q, 0.1f) * M_PI * 0.5f);
        sv->qnrm = sqrt(sv->q/2.0+0.01);
}

/* Change the frequency of a running SVF */

inline void setup_f_svf(sv_filter *sv, float fs, float fc) {
        sv->f = 2.0f * sin(M_PI * fc / ((float)(fs * F_R)));
}

/* Run one sample through the SV filter. Filter is by andy@vellocet */

inline float run_svf(sv_filter *sv, float in) {
        int i;

        in = sv->qnrm * in ;
        for (i=0; i < F_R; i++) {
                // only needed for pentium chips
                in  = DENORMALISE(in);
                sv->l = DENORMALISE(sv->l);
                // very slight waveshape for extra stability
                sv->b = sv->b - sv->b * sv->b * sv->b * 0.001f;

                // regular state variable code here
                // the notch and peaking outputs are optional
                sv->h = in - sv->l - sv->q * sv->b;
                sv->b = sv->b + sv->f * sv->h;
                sv->l = sv->l + sv->f * sv->b;

                in = sv->l;
        }

        return in;
}

#define ANALOGUEOSC_OSC_FREQ           0
#define ANALOGUEOSC_OSC_WAVE           1
#define ANALOGUEOSC_FM                 2
#define ANALOGUEOSC_OUTPUT             3

LADSPA_Descriptor *analogueOscDescriptor = NULL;

typedef struct {
	LADSPA_Data *osc_freq;
	LADSPA_Data *osc_wave;
	LADSPA_Data *fm;
	LADSPA_Data *output;
	sv_filter *  filter;
	float        osc_phase;
	LADSPA_Data run_adding_gain;
} AnalogueOsc;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return analogueOscDescriptor;
	default:
		return NULL;
	}
}

void activateAnalogueOsc(LADSPA_Handle instance) {
	AnalogueOsc *plugin_data = (AnalogueOsc *)instance;
	sv_filter *filter = plugin_data->filter;
	float osc_phase = plugin_data->osc_phase;
	osc_phase = 0;
	setup_svf(filter, sample_rate, 1000, F_Q);
	plugin_data->filter = filter;
	plugin_data->osc_phase = osc_phase;

}

void cleanupAnalogueOsc(LADSPA_Handle instance) {
	AnalogueOsc *plugin_data = (AnalogueOsc *)instance;
	free(plugin_data->filter);
	if (--tbl_ref_count == 0) {
	        free(sin_tbl);
	        free(tri_tbl);
	        free(squ_tbl);
	        free(saw_tbl);
	}
	free(instance);
}

void connectPortAnalogueOsc(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	AnalogueOsc *plugin;

	plugin = (AnalogueOsc *)instance;
	switch (port) {
	case ANALOGUEOSC_OSC_FREQ:
		plugin->osc_freq = data;
		break;
	case ANALOGUEOSC_OSC_WAVE:
		plugin->osc_wave = data;
		break;
	case ANALOGUEOSC_FM:
		plugin->fm = data;
		break;
	case ANALOGUEOSC_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateAnalogueOsc(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	AnalogueOsc *plugin_data = (AnalogueOsc *)malloc(sizeof(AnalogueOsc));
	sv_filter *filter = NULL;
	float osc_phase;

	long i;
	float tmp;
	
	sample_rate = s_rate;
	
	osc_phase = 0.0f;
	
	filter = calloc(1, sizeof(sv_filter));
	setup_svf(filter, sample_rate, 1000, F_Q);
	
	/* Build up denormalised oscilator wavetables, these are sample_rate
	   long, costs more RAM to create them but makes freqency calcs much
	   cheaper, and means that interpolation isn't that neccesary, esp if
	   you use integer frequncies */
	
	if (tbl_ref_count++ == 0) {
	        LADSPA_Data *tmp_tbl = malloc(sizeof(LADSPA_Data) * sample_rate);
	        const int lag = sample_rate/50;
	
	        sin_tbl = malloc(sizeof(LADSPA_Data) * sample_rate);
	        for (i = 0; i < sample_rate; i++) {
	                tmp = sin(i * 2 * M_PI / sample_rate);
	                sin_tbl[i] = DENORMALISE(tmp);
	        }
	
	        tri_tbl = malloc(sizeof(LADSPA_Data) * sample_rate);
	        for (i = 0; i < sample_rate; i++) {
	                tmp = acos(cos(i * 2 * M_PI / sample_rate)) / M_PI * 2 - 1;
	                tri_tbl[i] = DENORMALISE(tmp);
	        }
	
	        squ_tbl = malloc(sizeof(LADSPA_Data) * sample_rate);
	        for (i = 0; i < sample_rate/2; i++) {
	                tmp_tbl[i] = -1.0f;
	        }
	        for (i = sample_rate/2; i < sample_rate; i++) {
	                tmp_tbl[i] = +1.0f;
	        }
	        tmp = -1.0f;
	        for (i = (sample_rate/2)-lag; i < (sample_rate/2)+lag; i++) {
	                tmp_tbl[i] = tmp;
	                tmp += 1/(lag * 2);
	        }
	        for (i = 0; i < sample_rate; i++) {
	                squ_tbl[i] = (tmp_tbl[MOD(i-lag, sample_rate)] +
	                 tmp_tbl[MOD(i+lag, sample_rate)]) * 0.5;
	        }
	
	        saw_tbl = malloc(sizeof(LADSPA_Data) * sample_rate);
	        for (i = 0; i < sample_rate; i++) {
	                tmp = ((2.0 * i) - (float)sample_rate) / (float)sample_rate;
	                tmp_tbl[i] = DENORMALISE(tmp);
	        }
	        for (i = 0; i < sample_rate; i++) {
	                saw_tbl[i] = (tmp_tbl[MOD(i-lag, sample_rate)] +
	                 tmp_tbl[MOD(i+lag, sample_rate)]) * 0.5;
	        }
	
	        free(tmp_tbl);
	}

	plugin_data->filter = filter;
	plugin_data->osc_phase = osc_phase;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runAnalogueOsc(LADSPA_Handle instance, unsigned long sample_count) {
	AnalogueOsc *plugin_data = (AnalogueOsc *)instance;

	/* Osc freq (Hz) (float value) */
	LADSPA_Data osc_freq = *(plugin_data->osc_freq);

	/* Osc wave (0 = sin, 1 = tri, 2 = saw, 3 = squ) (float value) */
	LADSPA_Data osc_wave = *(plugin_data->osc_wave);

	/* FM input (Hz) (array of floats of length sample_count) */
	LADSPA_Data *fm = plugin_data->fm;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	sv_filter * filter = plugin_data->filter;
	float osc_phase = plugin_data->osc_phase;

	unsigned long pos;
	
	// Table for the osc to use
	float *osc_tbl, osc;
	osc_tbl = wave_tbl(osc_wave);
	
	// Calculate the step increment of the oscilator
	osc_freq = LIMIT(osc_freq, 0.0, sample_rate/16);
	setup_f_svf(filter, sample_rate, osc_freq * 16);
	
	for (pos = 0; pos < sample_count; pos++) {
	        osc = osc_tbl[(int)osc_phase];
	        osc_phase += osc_freq *
	         (1.0f + (rand() * 2*INST/(float)RAND_MAX - INST)) + fm[pos];
	        buffer_write(output[pos], run_svf(filter, osc));
	        while (osc_phase > sample_rate) {
	                osc_phase -= sample_rate;
	        }
	        while (osc_phase < 0) {
	                osc_phase += sample_rate;
	        }
	}
	plugin_data->osc_phase = osc_phase;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainAnalogueOsc(LADSPA_Handle instance, LADSPA_Data gain) {
	((AnalogueOsc *)instance)->run_adding_gain = gain;
}

void runAddingAnalogueOsc(LADSPA_Handle instance, unsigned long sample_count) {
	AnalogueOsc *plugin_data = (AnalogueOsc *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Osc freq (Hz) (float value) */
	LADSPA_Data osc_freq = *(plugin_data->osc_freq);

	/* Osc wave (0 = sin, 1 = tri, 2 = saw, 3 = squ) (float value) */
	LADSPA_Data osc_wave = *(plugin_data->osc_wave);

	/* FM input (Hz) (array of floats of length sample_count) */
	LADSPA_Data *fm = plugin_data->fm;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	sv_filter * filter = plugin_data->filter;
	float osc_phase = plugin_data->osc_phase;

	unsigned long pos;
	
	// Table for the osc to use
	float *osc_tbl, osc;
	osc_tbl = wave_tbl(osc_wave);
	
	// Calculate the step increment of the oscilator
	osc_freq = LIMIT(osc_freq, 0.0, sample_rate/16);
	setup_f_svf(filter, sample_rate, osc_freq * 16);
	
	for (pos = 0; pos < sample_count; pos++) {
	        osc = osc_tbl[(int)osc_phase];
	        osc_phase += osc_freq *
	         (1.0f + (rand() * 2*INST/(float)RAND_MAX - INST)) + fm[pos];
	        buffer_write(output[pos], run_svf(filter, osc));
	        while (osc_phase > sample_rate) {
	                osc_phase -= sample_rate;
	        }
	        while (osc_phase < 0) {
	                osc_phase += sample_rate;
	        }
	}
	plugin_data->osc_phase = osc_phase;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	analogueOscDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (analogueOscDescriptor) {
		analogueOscDescriptor->UniqueID = 1205;
		analogueOscDescriptor->Label = strdup("analogueOsc");
		analogueOscDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		analogueOscDescriptor->Name =
		 strdup("Analogue Oscillator");
		analogueOscDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		analogueOscDescriptor->Copyright =
		 strdup("GPL");
		analogueOscDescriptor->PortCount = 4;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(4,
		 sizeof(LADSPA_PortDescriptor));
		analogueOscDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(4,
		 sizeof(LADSPA_PortRangeHint));
		analogueOscDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(4, sizeof(char*));
		analogueOscDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Osc freq (Hz) */
		port_descriptors[ANALOGUEOSC_OSC_FREQ] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[ANALOGUEOSC_OSC_FREQ] =
		 strdup("Osc freq (Hz)");
		port_range_hints[ANALOGUEOSC_OSC_FREQ].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_SAMPLE_RATE;
		port_range_hints[ANALOGUEOSC_OSC_FREQ].LowerBound = 0;
		port_range_hints[ANALOGUEOSC_OSC_FREQ].UpperBound = 0.0313;

		/* Parameters for Osc wave (0 = sin, 1 = tri, 2 = saw, 3 = squ) */
		port_descriptors[ANALOGUEOSC_OSC_WAVE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[ANALOGUEOSC_OSC_WAVE] =
		 strdup("Osc wave (0 = sin, 1 = tri, 2 = saw, 3 = squ)");
		port_range_hints[ANALOGUEOSC_OSC_WAVE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_INTEGER;
		port_range_hints[ANALOGUEOSC_OSC_WAVE].LowerBound = 0;
		port_range_hints[ANALOGUEOSC_OSC_WAVE].UpperBound = 3;

		/* Parameters for FM input (Hz) */
		port_descriptors[ANALOGUEOSC_FM] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[ANALOGUEOSC_FM] =
		 strdup("FM input (Hz)");
		port_range_hints[ANALOGUEOSC_FM].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[ANALOGUEOSC_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[ANALOGUEOSC_OUTPUT] =
		 strdup("Output");
		port_range_hints[ANALOGUEOSC_OUTPUT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[ANALOGUEOSC_OUTPUT].LowerBound = -1;
		port_range_hints[ANALOGUEOSC_OUTPUT].UpperBound = +1;

		analogueOscDescriptor->activate = activateAnalogueOsc;
		analogueOscDescriptor->cleanup = cleanupAnalogueOsc;
		analogueOscDescriptor->connect_port = connectPortAnalogueOsc;
		analogueOscDescriptor->deactivate = NULL;
		analogueOscDescriptor->instantiate = instantiateAnalogueOsc;
		analogueOscDescriptor->run = runAnalogueOsc;
		analogueOscDescriptor->run_adding = runAddingAnalogueOsc;
		analogueOscDescriptor->set_run_adding_gain = setRunAddingGainAnalogueOsc;
	}
}

void _fini() {
	int i;

	if (analogueOscDescriptor) {
		free((char *)analogueOscDescriptor->Label);
		free((char *)analogueOscDescriptor->Name);
		free((char *)analogueOscDescriptor->Maker);
		free((char *)analogueOscDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)analogueOscDescriptor->PortDescriptors);
		for (i = 0; i < analogueOscDescriptor->PortCount; i++)
			free((char *)(analogueOscDescriptor->PortNames[i]));
		free((char **)analogueOscDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)analogueOscDescriptor->PortRangeHints);
		free(analogueOscDescriptor);
	}

}
