#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#define OVERDRIVE_DRIVE                0
#define OVERDRIVE_INPUT                1
#define OVERDRIVE_OUTPUT               2

LADSPA_Descriptor *overdriveDescriptor = NULL;

typedef struct {
	LADSPA_Data *drive;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data run_adding_gain;
} Overdrive;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return overdriveDescriptor;
	default:
		return NULL;
	}
}

void cleanupOverdrive(LADSPA_Handle instance) {
	free(instance);
}

void connectPortOverdrive(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Overdrive *plugin;

	plugin = (Overdrive *)instance;
	switch (port) {
	case OVERDRIVE_DRIVE:
		plugin->drive = data;
		break;
	case OVERDRIVE_INPUT:
		plugin->input = data;
		break;
	case OVERDRIVE_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateOverdrive(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Overdrive *plugin_data = (Overdrive *)malloc(sizeof(Overdrive));
	plugin_data->run_adding_gain = 1.0f;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runOverdrive(LADSPA_Handle instance, unsigned long sample_count) {
	Overdrive *plugin_data = (Overdrive *)instance;

	/* Drive level (float value) */
	LADSPA_Data drive = *(plugin_data->drive);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	unsigned long pos;

	for (pos = 0; pos < sample_count; pos++) {
	        LADSPA_Data x = *(input++);
	        buffer_write(*(output++), x*(fabs(x) + drive)/(x*x + (drive-1)*fabs(x) + 1));
	}
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainOverdrive(LADSPA_Handle instance, LADSPA_Data gain) {
	((Overdrive *)instance)->run_adding_gain = gain;
}

void runAddingOverdrive(LADSPA_Handle instance, unsigned long sample_count) {
	Overdrive *plugin_data = (Overdrive *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Drive level (float value) */
	LADSPA_Data drive = *(plugin_data->drive);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	unsigned long pos;

	for (pos = 0; pos < sample_count; pos++) {
	        LADSPA_Data x = *(input++);
	        buffer_write(*(output++), x*(fabs(x) + drive)/(x*x + (drive-1)*fabs(x) + 1));
	}
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	overdriveDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (overdriveDescriptor) {
		overdriveDescriptor->UniqueID = 1196;
		overdriveDescriptor->Label = strdup("overdrive");
		overdriveDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		overdriveDescriptor->Name =
		 strdup("Fast overdrive");
		overdriveDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		overdriveDescriptor->Copyright =
		 strdup("GPL");
		overdriveDescriptor->PortCount = 3;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(3,
		 sizeof(LADSPA_PortDescriptor));
		overdriveDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(3,
		 sizeof(LADSPA_PortRangeHint));
		overdriveDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(3, sizeof(char*));
		overdriveDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Drive level */
		port_descriptors[OVERDRIVE_DRIVE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[OVERDRIVE_DRIVE] =
		 strdup("Drive level");
		port_range_hints[OVERDRIVE_DRIVE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_DRIVE].LowerBound = 1;
		port_range_hints[OVERDRIVE_DRIVE].UpperBound = 3;

		/* Parameters for Input */
		port_descriptors[OVERDRIVE_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[OVERDRIVE_INPUT] =
		 strdup("Input");
		port_range_hints[OVERDRIVE_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[OVERDRIVE_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[OVERDRIVE_OUTPUT] =
		 strdup("Output");
		port_range_hints[OVERDRIVE_OUTPUT].HintDescriptor = 0;

		overdriveDescriptor->activate = NULL;
		overdriveDescriptor->cleanup = cleanupOverdrive;
		overdriveDescriptor->connect_port = connectPortOverdrive;
		overdriveDescriptor->deactivate = NULL;
		overdriveDescriptor->instantiate = instantiateOverdrive;
		overdriveDescriptor->run = runOverdrive;
		overdriveDescriptor->run_adding = runAddingOverdrive;
		overdriveDescriptor->set_run_adding_gain = setRunAddingGainOverdrive;
	}
}

void _fini() {
	int i;

	if (overdriveDescriptor) {
		free((char *)overdriveDescriptor->Label);
		free((char *)overdriveDescriptor->Name);
		free((char *)overdriveDescriptor->Maker);
		free((char *)overdriveDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)overdriveDescriptor->PortDescriptors);
		for (i = 0; i < overdriveDescriptor->PortCount; i++)
			free((char *)(overdriveDescriptor->PortNames[i]));
		free((char **)overdriveDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)overdriveDescriptor->PortRangeHints);
		free(overdriveDescriptor);
	}

}
