/*

        Copyright (C) 1999 Juhana Sadeharju
                       kouhia at nic.funet.fi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "gverbdsp.h"
#include "gverb.h"

ty_gverb *gverb_new(int srate, float maxroomsize, float roomsize,
		    float revtime,
		    float damping, float spread,
		    float inputbandwidth, float earlylevel,
		    float taillevel)
{
  ty_gverb *p;
  float ga,gb,gt;
  int i,n;
  float r;
  float diffscale;
  int a,b,c,cc,d,dd,e;
  float spread1,spread2;

  p = (ty_gverb *)malloc(sizeof(ty_gverb));
  p->rate = srate;
  p->fdndamping = damping;
  p->maxroomsize = maxroomsize;
  p->roomsize = roomsize;
  p->revtime = revtime;
  p->earlylevel = earlylevel;
  p->taillevel = taillevel;

  p->maxdelay = p->rate*p->maxroomsize/340.0;
  p->largestdelay = p->rate*p->roomsize/340.0;


  /* Input damper */

  p->inputbandwidth = inputbandwidth;
  p->inputdamper = damper_make(1.0 - p->inputbandwidth);


  /* FDN section */


  p->fdndels = (ty_fixeddelay **)malloc(FDNORDER*sizeof(ty_fixeddelay *));
  for(i = 0; i < FDNORDER; i++) {
    p->fdndels[i] = fixeddelay_make((int)p->maxdelay+1000);
  }
  p->fdngains = (float *)malloc(FDNORDER*sizeof(float));
  p->fdnlens = (int *)malloc(FDNORDER*sizeof(int));

  p->fdndamps = (ty_damper **)malloc(FDNORDER*sizeof(ty_damper *));
  for(i = 0; i < FDNORDER; i++) {
    p->fdndamps[i] = damper_make(p->fdndamping);
  }

  ga = 60.0;
  gt = p->revtime;
  ga = pow(10.0,-ga/20.0);
  n = p->rate*gt;
  p->alpha = pow((double)ga,(double)1.0/(double)n);

  gb = 0.0;
  for(i = 0; i < FDNORDER; i++) {
    if (i == 0) gb = 1.000000*p->largestdelay;
    if (i == 1) gb = 0.816490*p->largestdelay;
    if (i == 2) gb = 0.707100*p->largestdelay;
    if (i == 3) gb = 0.632450*p->largestdelay;

#if 0
    p->fdnlens[i] = nearest_prime((int)gb, 0.5);
#else
    p->fdnlens[i] = (int)gb;
#endif
    p->fdngains[i] = -pow(p->alpha,(double)p->fdnlens[i]);
  }

  p->d = (float *)malloc(FDNORDER*sizeof(float));
  p->u = (float *)malloc(FDNORDER*sizeof(float));
  p->f = (float *)malloc(FDNORDER*sizeof(float));



  /* Diffuser section */


  diffscale = (float)p->fdnlens[3]/(210+159+562+410);
  spread1 = spread;
  spread2 = 3.0*spread;

  b = 210;
  r = 0.125541;
  a = spread1*r;
  c = 210+159+a;
  cc = c-b;
  r = 0.854046;
  a = spread2*r;
  d = 210+159+562+a;
  dd = d-c;
  e = 1341-d;

  p->ldifs = (ty_diffuser **)malloc(4*sizeof(ty_diffuser *));
  p->ldifs[0] = diffuser_make((int)(diffscale*b),0.75);
  p->ldifs[1] = diffuser_make((int)(diffscale*cc),0.75);
  p->ldifs[2] = diffuser_make((int)(diffscale*dd),0.625);
  p->ldifs[3] = diffuser_make((int)(diffscale*e),0.625);

  b = 210;
  r = -0.568366;
  a = spread1*r;
  c = 210+159+a;
  cc = c-b;
  r = -0.126815;
  a = spread2*r;
  d = 210+159+562+a;
  dd = d-c;
  e = 1341-d;

  p->rdifs = (ty_diffuser **)malloc(4*sizeof(ty_diffuser *));
  p->rdifs[0] = diffuser_make((int)(diffscale*b),0.75);
  p->rdifs[1] = diffuser_make((int)(diffscale*cc),0.75);
  p->rdifs[2] = diffuser_make((int)(diffscale*dd),0.625);
  p->rdifs[3] = diffuser_make((int)(diffscale*e),0.625);



  /* Tapped delay section */

  p->tapdelay = fixeddelay_make(44000);
  p->taps = (int *)malloc(FDNORDER*sizeof(int));
  p->tapgains = (float *)malloc(FDNORDER*sizeof(float));

  p->taps[0] = 5+0.410*p->largestdelay;
  p->taps[1] = 5+0.300*p->largestdelay;
  p->taps[2] = 5+0.155*p->largestdelay;
  p->taps[3] = 5+0.000*p->largestdelay;

  for(i = 0; i < FDNORDER; i++) {
    p->tapgains[i] = pow(p->alpha,(double)p->taps[i]);
  }

  return(p);
}

void gverb_free(ty_gverb *p)
{
  int i;

  damper_free(p->inputdamper);
  for(i = 0; i < FDNORDER; i++) {
    fixeddelay_free(p->fdndels[i]);
    damper_free(p->fdndamps[i]);
    diffuser_free(p->ldifs[i]);
    diffuser_free(p->rdifs[i]);
  }
  free(p->fdndels);
  free(p->fdngains);
  free(p->fdnlens);
  free(p->fdndamps);
  free(p->d);
  free(p->u);
  free(p->f);
  free(p->ldifs);
  free(p->rdifs);
  free(p->taps);
  free(p->tapgains);
  fixeddelay_free(p->tapdelay);
}

inline void gverb_fdnmatrix(float *a, float *b)
{
  const float dl0 = a[0], dl1 = a[1], dl2 = a[2], dl3 = a[3];

  b[0] = 0.5*(+dl0 + dl1 - dl2 - dl3);
  b[1] = 0.5*(+dl0 - dl1 - dl2 + dl3);
  b[2] = 0.5*(-dl0 + dl1 - dl2 + dl3);
  b[3] = 0.5*(+dl0 + dl1 + dl2 + dl3);
}

inline void gverb_do(ty_gverb *p, float x, float *yl, float *yr)
{
  float z;
  int i;
  float lsum,rsum,sum,sign;

  z = damper_do(p->inputdamper,x);

  z = diffuser_do(p->ldifs[0],z);

  for(i = 0; i < FDNORDER; i++) {
    p->u[i] = p->tapgains[i]*fixeddelay_read(p->tapdelay,p->taps[i]);
  }
  fixeddelay_write(p->tapdelay,z);

  for(i = 0; i < FDNORDER; i++) {
    p->d[i] = damper_do(p->fdndamps[i],
			p->fdngains[i]*fixeddelay_read(p->fdndels[i],
						       p->fdnlens[i]));
  }

  sum = 0.0f;
  sign = 1.0f;
  for(i = 0; i < FDNORDER; i++) {
    sum += sign*(p->taillevel*p->d[i] + p->earlylevel*p->u[i]);
    sign = -sign;
  }
  sum += x*p->earlylevel;
  lsum = sum;
  rsum = sum;

  gverb_fdnmatrix(p->d,p->f);

  for(i = 0; i < FDNORDER; i++) {
    fixeddelay_write(p->fdndels[i],p->u[i]+p->f[i]);
  }

  lsum = diffuser_do(p->ldifs[1],lsum);
  lsum = diffuser_do(p->ldifs[2],lsum);
  lsum = diffuser_do(p->ldifs[3],lsum);
  rsum = diffuser_do(p->rdifs[1],rsum);
  rsum = diffuser_do(p->rdifs[2],rsum);
  rsum = diffuser_do(p->rdifs[3],rsum);

  *yl = lsum;
  *yr = rsum;
}

inline void gverb_set_roomsize(ty_gverb *p,float a)
{
  float gb;
  int i;

  p->roomsize = a;
  p->largestdelay = p->rate*p->roomsize*0.0029411764f;

  gb = 0.0;
  for(i = 0; i < FDNORDER; i++) {
    if (i == 0) gb = 1.000000f*p->largestdelay;
    if (i == 1) gb = 0.816490f*p->largestdelay;
    if (i == 2) gb = 0.707100f*p->largestdelay;
    if (i == 3) gb = 0.632450f*p->largestdelay;

#if 0
    p->fdnlens[i] = nearest_prime(f_round(gb), 0.5);
#else
    p->fdnlens[i] = ff_round(gb);
#endif
    p->fdngains[i] = -pow(p->alpha,(double)p->fdnlens[i]);
  }

}

inline void gverb_set_revtime(ty_gverb *p,float a)
{
  float ga,gt;
  double n;
  int i;

  p->revtime = a;

  ga = 60.0;
  gt = p->revtime;
  ga = pow(10.0,-ga/20.0);
  n = p->rate*gt;
  p->alpha = pow((double)ga,(double)1.0/n);

  for(i = 0; i < FDNORDER; i++) {
    p->fdngains[i] = -pow(p->alpha,(double)p->fdnlens[i]);
  }

}

inline void gverb_set_damping(ty_gverb *p,float a)
{
  int i;

  p->fdndamping = a;
  for(i = 0; i < FDNORDER; i++) {
    damper_set(p->fdndamps[i],p->fdndamping);
  }
}

inline void gverb_set_inputbandwidth(ty_gverb *p,float a)
{
  p->inputbandwidth = a;
  damper_set(p->inputdamper,1.0 - p->inputbandwidth);
}

inline void gverb_set_earlylevel(ty_gverb *p,float a)
{
  p->earlylevel = a;
}

inline void gverb_set_taillevel(ty_gverb *p,float a)
{
  p->taillevel = a;
}

