#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#include "ladspa-util.h"
#define MAX_LAWS 7

inline void rc_window(float *window, float *meta_window, int law_p, int max_law_p) {
        long i;
        float step, track = 0.0f;

        step = (float)max_law_p / (float)law_p;

        // Fill window with a raised cosine function
        for (i=0; i<law_p; i++) {
                window[i] = meta_window[f_round(track)];
                window[law_p * 2 - i] = window[i];
                track += step;
        }
        window[law_p] = 1.0f;
}

#define MULTIVOICECHORUS_VOICES        0
#define MULTIVOICECHORUS_DELAY_BASE    1
#define MULTIVOICECHORUS_VOICE_SPREAD  2
#define MULTIVOICECHORUS_DETUNE        3
#define MULTIVOICECHORUS_LAW_FREQ      4
#define MULTIVOICECHORUS_ATTEN         5
#define MULTIVOICECHORUS_INPUT         6
#define MULTIVOICECHORUS_OUTPUT        7

LADSPA_Descriptor *multivoiceChorusDescriptor = NULL;

typedef struct {
	LADSPA_Data *voices;
	LADSPA_Data *delay_base;
	LADSPA_Data *voice_spread;
	LADSPA_Data *detune;
	LADSPA_Data *law_freq;
	LADSPA_Data *atten;
	LADSPA_Data *input;
	LADSPA_Data *output;
	long         count;
	int          delay_pos;
	int          delay_size;
	float *      delay_tbl;
	int          last_law_p;
	int          law_pos;
	int          law_roll;
	float **     law_tbl;
	int          max_law_p;
	float *      meta_window;
	long         sample_rate;
	float *      window;
	LADSPA_Data run_adding_gain;
} MultivoiceChorus;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return multivoiceChorusDescriptor;
	default:
		return NULL;
	}
}

void cleanupMultivoiceChorus(LADSPA_Handle instance) {
	MultivoiceChorus *plugin_data = (MultivoiceChorus *)instance;
	int i;
	
	for (i=0; i<MAX_LAWS; i++) {
	        free(plugin_data->law_tbl[i]);
	}
	free(plugin_data->law_tbl);
	free(plugin_data->delay_tbl);
	free(plugin_data->window);
	free(instance);
}

void connectPortMultivoiceChorus(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	MultivoiceChorus *plugin;

	plugin = (MultivoiceChorus *)instance;
	switch (port) {
	case MULTIVOICECHORUS_VOICES:
		plugin->voices = data;
		break;
	case MULTIVOICECHORUS_DELAY_BASE:
		plugin->delay_base = data;
		break;
	case MULTIVOICECHORUS_VOICE_SPREAD:
		plugin->voice_spread = data;
		break;
	case MULTIVOICECHORUS_DETUNE:
		plugin->detune = data;
		break;
	case MULTIVOICECHORUS_LAW_FREQ:
		plugin->law_freq = data;
		break;
	case MULTIVOICECHORUS_ATTEN:
		plugin->atten = data;
		break;
	case MULTIVOICECHORUS_INPUT:
		plugin->input = data;
		break;
	case MULTIVOICECHORUS_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateMultivoiceChorus(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	MultivoiceChorus *plugin_data = (MultivoiceChorus *)malloc(sizeof(MultivoiceChorus));
	long count;
	int delay_pos;
	int delay_size;
	float *delay_tbl = NULL;
	int last_law_p;
	int law_pos;
	int law_roll;
	float **law_tbl = NULL;
	int max_law_p;
	float *meta_window = NULL;
	long sample_rate;
	float *window = NULL;

	int i;
	
	sample_rate = s_rate;
	
	max_law_p = s_rate/2;
	last_law_p = -1;
	law_tbl = malloc(sizeof(float *) * MAX_LAWS);
	for (i=0; i<MAX_LAWS; i++) {
	        law_tbl[i] = calloc(sizeof(float), max_law_p * 2);
	}
	law_pos = 0;
	law_roll = 0;
	
	meta_window = calloc(sizeof(float), max_law_p);
	window = calloc(sizeof(float), max_law_p * 2);
	for (i=0; i<max_law_p; i++) {
	        meta_window[i] = cos((((float)i - (float)max_law_p) / (float)(max_law_p * 2)) * M_PI);
	        meta_window[i] *= meta_window[i];
	}
	
	delay_size = sample_rate / 10;
	delay_tbl = calloc(sizeof(float), delay_size);
	delay_pos = 0;
	
	count = 0;

	plugin_data->count = count;
	plugin_data->delay_pos = delay_pos;
	plugin_data->delay_size = delay_size;
	plugin_data->delay_tbl = delay_tbl;
	plugin_data->last_law_p = last_law_p;
	plugin_data->law_pos = law_pos;
	plugin_data->law_roll = law_roll;
	plugin_data->law_tbl = law_tbl;
	plugin_data->max_law_p = max_law_p;
	plugin_data->meta_window = meta_window;
	plugin_data->sample_rate = sample_rate;
	plugin_data->window = window;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runMultivoiceChorus(LADSPA_Handle instance, unsigned long sample_count) {
	MultivoiceChorus *plugin_data = (MultivoiceChorus *)instance;

	/* Number of voices (float value) */
	LADSPA_Data voices = *(plugin_data->voices);

	/* Delay base (ms) (float value) */
	LADSPA_Data delay_base = *(plugin_data->delay_base);

	/* Voice separation (ms) (float value) */
	LADSPA_Data voice_spread = *(plugin_data->voice_spread);

	/* Detune (%) (float value) */
	LADSPA_Data detune = *(plugin_data->detune);

	/* LFO frequency (Hz) (float value) */
	LADSPA_Data law_freq = *(plugin_data->law_freq);

	/* Output attenuation (dB) (float value) */
	LADSPA_Data atten = *(plugin_data->atten);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	long count = plugin_data->count;
	int delay_pos = plugin_data->delay_pos;
	int delay_size = plugin_data->delay_size;
	float * delay_tbl = plugin_data->delay_tbl;
	int last_law_p = plugin_data->last_law_p;
	int law_pos = plugin_data->law_pos;
	int law_roll = plugin_data->law_roll;
	float ** law_tbl = plugin_data->law_tbl;
	int max_law_p = plugin_data->max_law_p;
	float * meta_window = plugin_data->meta_window;
	long sample_rate = plugin_data->sample_rate;
	float * window = plugin_data->window;

	unsigned long pos;
	int i, d_base, t;
	LADSPA_Data out;
	float delay_depth;
	float dp; // float delay position
	float dp_frac; // fractional part
	int dp_idx; // Integer delay index
	int laws, law_separation, base_offset;
	int law_p; // Period of law
	
	// Set law params
	laws = LIMIT(voices-1, 0, 7);
	law_p = LIMIT((1.0f/law_freq) * sample_rate, 1, max_law_p);
	if (laws > 0) {
	        law_separation = law_p / laws;
	} else {
	        law_separation = 0;
	}
	
	// Calculate voice spread in samples
	base_offset = (voice_spread * sample_rate) / 1000;
	// Calculate base delay size in samples
	d_base = (delay_base * sample_rate) / 1000;
	// Calculate delay depth in samples
	delay_depth = LIMIT((law_p * detune) / (100.0f * M_PI), 0, delay_size - d_base - 1 - (base_offset * laws));
	
	// Calculate output attenuation
	atten = DB_CO(atten);
	
	for (pos = 0; pos < sample_count; pos++) {
	        // N times per law 'frequency' splurge a new set of windowed data
	        // into one of the N law buffers. Keeps the laws out of phase.
	        if (laws > 0 && (count % law_separation) == 0) {
	                float n = (float)rand() / (float)RAND_MAX;
	
	                // Is the law table up to date?
	                if (last_law_p != law_p) {
	                        rc_window(window, meta_window, law_p, max_law_p);
	                        last_law_p = law_p;
	                }
	
	                // Pick the next law to be changed
	                law_roll = (law_roll + 1) % laws;
	
	                // Interpolate into the chosen law table
	                for (i = 0; i<law_p*2; i++) {
	                        law_tbl[law_roll][(i+law_pos) % (max_law_p*2)] += window[i] * n;
	                }
	        }
	
	        out = input[pos];
	        for (t=0; t<laws; t++) {
	                // Calculate position in delay table
	                dp = (float)(delay_pos - d_base + (t*base_offset)) - (delay_depth * law_tbl[t][law_pos]);
	                // Get the integer part
	                dp_idx = (int)dp;
	                // Get the fractional part
	                dp_frac = dp - dp_idx;
	                // Calculate the modulo'd table index
	                dp_idx = MOD(dp_idx, delay_size);
	
	                // Zero the used law buffer
	                law_tbl[t][law_pos] = 0.0f;
	
	                // Accumulate into output buffer
	                out += cube_interp(dp_frac, delay_tbl[MOD(dp_idx-1, delay_size)], delay_tbl[dp_idx], delay_tbl[(dp_idx+1) % delay_size], delay_tbl[(dp_idx+2) % delay_size]);
	        }
	        law_pos = (law_pos + 1) % (max_law_p * 2);
	
	        // Store new delay value
	        delay_tbl[delay_pos] = input[pos];
	        delay_pos = (delay_pos + 1) % delay_size;
	
	        buffer_write(output[pos], out * atten);
	        count++;
	}
	
	plugin_data->count = count;
	plugin_data->law_pos = law_pos;
	plugin_data->last_law_p = last_law_p;
	plugin_data->law_roll = law_roll;
	plugin_data->delay_pos = delay_pos;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainMultivoiceChorus(LADSPA_Handle instance, LADSPA_Data gain) {
	((MultivoiceChorus *)instance)->run_adding_gain = gain;
}

void runAddingMultivoiceChorus(LADSPA_Handle instance, unsigned long sample_count) {
	MultivoiceChorus *plugin_data = (MultivoiceChorus *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Number of voices (float value) */
	LADSPA_Data voices = *(plugin_data->voices);

	/* Delay base (ms) (float value) */
	LADSPA_Data delay_base = *(plugin_data->delay_base);

	/* Voice separation (ms) (float value) */
	LADSPA_Data voice_spread = *(plugin_data->voice_spread);

	/* Detune (%) (float value) */
	LADSPA_Data detune = *(plugin_data->detune);

	/* LFO frequency (Hz) (float value) */
	LADSPA_Data law_freq = *(plugin_data->law_freq);

	/* Output attenuation (dB) (float value) */
	LADSPA_Data atten = *(plugin_data->atten);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	long count = plugin_data->count;
	int delay_pos = plugin_data->delay_pos;
	int delay_size = plugin_data->delay_size;
	float * delay_tbl = plugin_data->delay_tbl;
	int last_law_p = plugin_data->last_law_p;
	int law_pos = plugin_data->law_pos;
	int law_roll = plugin_data->law_roll;
	float ** law_tbl = plugin_data->law_tbl;
	int max_law_p = plugin_data->max_law_p;
	float * meta_window = plugin_data->meta_window;
	long sample_rate = plugin_data->sample_rate;
	float * window = plugin_data->window;

	unsigned long pos;
	int i, d_base, t;
	LADSPA_Data out;
	float delay_depth;
	float dp; // float delay position
	float dp_frac; // fractional part
	int dp_idx; // Integer delay index
	int laws, law_separation, base_offset;
	int law_p; // Period of law
	
	// Set law params
	laws = LIMIT(voices-1, 0, 7);
	law_p = LIMIT((1.0f/law_freq) * sample_rate, 1, max_law_p);
	if (laws > 0) {
	        law_separation = law_p / laws;
	} else {
	        law_separation = 0;
	}
	
	// Calculate voice spread in samples
	base_offset = (voice_spread * sample_rate) / 1000;
	// Calculate base delay size in samples
	d_base = (delay_base * sample_rate) / 1000;
	// Calculate delay depth in samples
	delay_depth = LIMIT((law_p * detune) / (100.0f * M_PI), 0, delay_size - d_base - 1 - (base_offset * laws));
	
	// Calculate output attenuation
	atten = DB_CO(atten);
	
	for (pos = 0; pos < sample_count; pos++) {
	        // N times per law 'frequency' splurge a new set of windowed data
	        // into one of the N law buffers. Keeps the laws out of phase.
	        if (laws > 0 && (count % law_separation) == 0) {
	                float n = (float)rand() / (float)RAND_MAX;
	
	                // Is the law table up to date?
	                if (last_law_p != law_p) {
	                        rc_window(window, meta_window, law_p, max_law_p);
	                        last_law_p = law_p;
	                }
	
	                // Pick the next law to be changed
	                law_roll = (law_roll + 1) % laws;
	
	                // Interpolate into the chosen law table
	                for (i = 0; i<law_p*2; i++) {
	                        law_tbl[law_roll][(i+law_pos) % (max_law_p*2)] += window[i] * n;
	                }
	        }
	
	        out = input[pos];
	        for (t=0; t<laws; t++) {
	                // Calculate position in delay table
	                dp = (float)(delay_pos - d_base + (t*base_offset)) - (delay_depth * law_tbl[t][law_pos]);
	                // Get the integer part
	                dp_idx = (int)dp;
	                // Get the fractional part
	                dp_frac = dp - dp_idx;
	                // Calculate the modulo'd table index
	                dp_idx = MOD(dp_idx, delay_size);
	
	                // Zero the used law buffer
	                law_tbl[t][law_pos] = 0.0f;
	
	                // Accumulate into output buffer
	                out += cube_interp(dp_frac, delay_tbl[MOD(dp_idx-1, delay_size)], delay_tbl[dp_idx], delay_tbl[(dp_idx+1) % delay_size], delay_tbl[(dp_idx+2) % delay_size]);
	        }
	        law_pos = (law_pos + 1) % (max_law_p * 2);
	
	        // Store new delay value
	        delay_tbl[delay_pos] = input[pos];
	        delay_pos = (delay_pos + 1) % delay_size;
	
	        buffer_write(output[pos], out * atten);
	        count++;
	}
	
	plugin_data->count = count;
	plugin_data->law_pos = law_pos;
	plugin_data->last_law_p = last_law_p;
	plugin_data->law_roll = law_roll;
	plugin_data->delay_pos = delay_pos;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	multivoiceChorusDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (multivoiceChorusDescriptor) {
		multivoiceChorusDescriptor->UniqueID = 1201;
		multivoiceChorusDescriptor->Label = strdup("multivoiceChorus");
		multivoiceChorusDescriptor->Properties =
		 0;
		multivoiceChorusDescriptor->Name =
		 strdup("Multivoice Chorus");
		multivoiceChorusDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		multivoiceChorusDescriptor->Copyright =
		 strdup("GPL");
		multivoiceChorusDescriptor->PortCount = 8;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(8,
		 sizeof(LADSPA_PortDescriptor));
		multivoiceChorusDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(8,
		 sizeof(LADSPA_PortRangeHint));
		multivoiceChorusDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(8, sizeof(char*));
		multivoiceChorusDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Number of voices */
		port_descriptors[MULTIVOICECHORUS_VOICES] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[MULTIVOICECHORUS_VOICES] =
		 strdup("Number of voices");
		port_range_hints[MULTIVOICECHORUS_VOICES].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE | LADSPA_HINT_INTEGER;
		port_range_hints[MULTIVOICECHORUS_VOICES].LowerBound = 1;
		port_range_hints[MULTIVOICECHORUS_VOICES].UpperBound = 8;

		/* Parameters for Delay base (ms) */
		port_descriptors[MULTIVOICECHORUS_DELAY_BASE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[MULTIVOICECHORUS_DELAY_BASE] =
		 strdup("Delay base (ms)");
		port_range_hints[MULTIVOICECHORUS_DELAY_BASE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[MULTIVOICECHORUS_DELAY_BASE].LowerBound = 10;
		port_range_hints[MULTIVOICECHORUS_DELAY_BASE].UpperBound = 40;

		/* Parameters for Voice separation (ms) */
		port_descriptors[MULTIVOICECHORUS_VOICE_SPREAD] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[MULTIVOICECHORUS_VOICE_SPREAD] =
		 strdup("Voice separation (ms)");
		port_range_hints[MULTIVOICECHORUS_VOICE_SPREAD].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[MULTIVOICECHORUS_VOICE_SPREAD].LowerBound = 0;
		port_range_hints[MULTIVOICECHORUS_VOICE_SPREAD].UpperBound = 2;

		/* Parameters for Detune (%) */
		port_descriptors[MULTIVOICECHORUS_DETUNE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[MULTIVOICECHORUS_DETUNE] =
		 strdup("Detune (%)");
		port_range_hints[MULTIVOICECHORUS_DETUNE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[MULTIVOICECHORUS_DETUNE].LowerBound = 0;
		port_range_hints[MULTIVOICECHORUS_DETUNE].UpperBound = 5;

		/* Parameters for LFO frequency (Hz) */
		port_descriptors[MULTIVOICECHORUS_LAW_FREQ] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[MULTIVOICECHORUS_LAW_FREQ] =
		 strdup("LFO frequency (Hz)");
		port_range_hints[MULTIVOICECHORUS_LAW_FREQ].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[MULTIVOICECHORUS_LAW_FREQ].LowerBound = 2;
		port_range_hints[MULTIVOICECHORUS_LAW_FREQ].UpperBound = 30;

		/* Parameters for Output attenuation (dB) */
		port_descriptors[MULTIVOICECHORUS_ATTEN] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[MULTIVOICECHORUS_ATTEN] =
		 strdup("Output attenuation (dB)");
		port_range_hints[MULTIVOICECHORUS_ATTEN].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[MULTIVOICECHORUS_ATTEN].LowerBound = -20;
		port_range_hints[MULTIVOICECHORUS_ATTEN].UpperBound = 0;

		/* Parameters for Input */
		port_descriptors[MULTIVOICECHORUS_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[MULTIVOICECHORUS_INPUT] =
		 strdup("Input");
		port_range_hints[MULTIVOICECHORUS_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[MULTIVOICECHORUS_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[MULTIVOICECHORUS_OUTPUT] =
		 strdup("Output");
		port_range_hints[MULTIVOICECHORUS_OUTPUT].HintDescriptor = 0;

		multivoiceChorusDescriptor->activate = NULL;
		multivoiceChorusDescriptor->cleanup = cleanupMultivoiceChorus;
		multivoiceChorusDescriptor->connect_port = connectPortMultivoiceChorus;
		multivoiceChorusDescriptor->deactivate = NULL;
		multivoiceChorusDescriptor->instantiate = instantiateMultivoiceChorus;
		multivoiceChorusDescriptor->run = runMultivoiceChorus;
		multivoiceChorusDescriptor->run_adding = runAddingMultivoiceChorus;
		multivoiceChorusDescriptor->set_run_adding_gain = setRunAddingGainMultivoiceChorus;
	}
}

void _fini() {
	int i;

	if (multivoiceChorusDescriptor) {
		free((char *)multivoiceChorusDescriptor->Label);
		free((char *)multivoiceChorusDescriptor->Name);
		free((char *)multivoiceChorusDescriptor->Maker);
		free((char *)multivoiceChorusDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)multivoiceChorusDescriptor->PortDescriptors);
		for (i = 0; i < multivoiceChorusDescriptor->PortCount; i++)
			free((char *)(multivoiceChorusDescriptor->PortNames[i]));
		free((char **)multivoiceChorusDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)multivoiceChorusDescriptor->PortRangeHints);
		free(multivoiceChorusDescriptor);
	}

}
