#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#include "ladspa-util.h"

#define BASE_BUFFER 0.001 // Base buffer length (s)

inline void rc_window(float *window, float *meta_window, long law_p, long max_law_p) {
        long i;
        float step, track = 0.0f;

        step = (float)max_law_p / (float)law_p;

        // Fill window with a raised cosine function
        for (i=0; i<law_p; i++) {
                window[i] = meta_window[f_round(track)];
                window[law_p * 2 - i] = window[i];
                track += step;
        }
        window[law_p] = 1.0f;
}

inline LADSPA_Data sat(LADSPA_Data x, float q,  float dist) {
        if (x == q) {
                return 1.0f / dist + q / (1.0f - exp(dist * q));
        }
        return ((x - q) / (1.0f - exp(-dist * (x - q))) + q /
         (1.0f - exp(dist * q)));
}

#define RETROFLANGE_DELAY_DEPTH        0
#define RETROFLANGE_LAW_FREQ           1
#define RETROFLANGE_INPUT              2
#define RETROFLANGE_OUTPUT             3

LADSPA_Descriptor *retroFlangeDescriptor = NULL;

typedef struct {
	LADSPA_Data *delay_depth;
	LADSPA_Data *law_freq;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data *buffer;
	long         buffer_size;
	long         count;
	LADSPA_Data *delay_line;
	int          delay_line_length;
	int          delay_pos;
	LADSPA_Data  last_in;
	int          last_law_p;
	int          last_phase;
	int          law_pos;
	float *      law_tbl;
	int          max_law_p;
	float *      meta_window;
	float        phase;
	long         sample_rate;
	float *      window;
	LADSPA_Data  z0;
	LADSPA_Data  z1;
	LADSPA_Data  z2;
	LADSPA_Data run_adding_gain;
} RetroFlange;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return retroFlangeDescriptor;
	default:
		return NULL;
	}
}

void cleanupRetroFlange(LADSPA_Handle instance) {
	free(instance);
}

void connectPortRetroFlange(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	RetroFlange *plugin;

	plugin = (RetroFlange *)instance;
	switch (port) {
	case RETROFLANGE_DELAY_DEPTH:
		plugin->delay_depth = data;
		break;
	case RETROFLANGE_LAW_FREQ:
		plugin->law_freq = data;
		break;
	case RETROFLANGE_INPUT:
		plugin->input = data;
		break;
	case RETROFLANGE_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateRetroFlange(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	RetroFlange *plugin_data = (RetroFlange *)malloc(sizeof(RetroFlange));
	LADSPA_Data *buffer = NULL;
	long buffer_size;
	long count;
	LADSPA_Data *delay_line = NULL;
	int delay_line_length;
	int delay_pos;
	LADSPA_Data last_in;
	int last_law_p;
	int last_phase;
	int law_pos;
	float *law_tbl = NULL;
	int max_law_p;
	float *meta_window = NULL;
	float phase;
	long sample_rate;
	float *window = NULL;
	LADSPA_Data z0;
	LADSPA_Data z1;
	LADSPA_Data z2;

	int i;
	
	sample_rate = s_rate;
	buffer_size = BASE_BUFFER * s_rate;
	buffer = calloc(buffer_size, sizeof(LADSPA_Data));
	phase = 0;
	last_phase = 0;
	last_in = 0.0f;
	max_law_p = s_rate*2;
	last_law_p = -1;
	law_tbl = calloc(sizeof(float), max_law_p * 2);
	law_pos = 0;
	delay_line_length =  sample_rate * 0.01f;
	delay_line = calloc(sizeof(float), delay_line_length);
	
	meta_window = calloc(sizeof(float), max_law_p);
	window = calloc(sizeof(float), max_law_p * 2);
	for (i=0; i<max_law_p; i++) {
	        meta_window[i] = cos((((float)i - (float)max_law_p) / (float)(max_law_p * 2)) * M_PI);
	        meta_window[i] *= meta_window[i];
	}
	
	delay_pos = 0;
	count = 0;
	
	z0 = 0.0f;
	z1 = 0.0f;
	z2 = 0.0f;

	plugin_data->buffer = buffer;
	plugin_data->buffer_size = buffer_size;
	plugin_data->count = count;
	plugin_data->delay_line = delay_line;
	plugin_data->delay_line_length = delay_line_length;
	plugin_data->delay_pos = delay_pos;
	plugin_data->last_in = last_in;
	plugin_data->last_law_p = last_law_p;
	plugin_data->last_phase = last_phase;
	plugin_data->law_pos = law_pos;
	plugin_data->law_tbl = law_tbl;
	plugin_data->max_law_p = max_law_p;
	plugin_data->meta_window = meta_window;
	plugin_data->phase = phase;
	plugin_data->sample_rate = sample_rate;
	plugin_data->window = window;
	plugin_data->z0 = z0;
	plugin_data->z1 = z1;
	plugin_data->z2 = z2;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runRetroFlange(LADSPA_Handle instance, unsigned long sample_count) {
	RetroFlange *plugin_data = (RetroFlange *)instance;

	/* Average stall (ms) (float value) */
	LADSPA_Data delay_depth = *(plugin_data->delay_depth);

	/* Flange frequency (Hz) (float value) */
	LADSPA_Data law_freq = *(plugin_data->law_freq);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data * buffer = plugin_data->buffer;
	long buffer_size = plugin_data->buffer_size;
	long count = plugin_data->count;
	LADSPA_Data * delay_line = plugin_data->delay_line;
	int delay_line_length = plugin_data->delay_line_length;
	int delay_pos = plugin_data->delay_pos;
	LADSPA_Data last_in = plugin_data->last_in;
	int last_law_p = plugin_data->last_law_p;
	int last_phase = plugin_data->last_phase;
	int law_pos = plugin_data->law_pos;
	float * law_tbl = plugin_data->law_tbl;
	int max_law_p = plugin_data->max_law_p;
	float * meta_window = plugin_data->meta_window;
	float phase = plugin_data->phase;
	long sample_rate = plugin_data->sample_rate;
	float * window = plugin_data->window;
	LADSPA_Data z0 = plugin_data->z0;
	LADSPA_Data z1 = plugin_data->z1;
	LADSPA_Data z2 = plugin_data->z2;

	long int pos;
	int law_p = f_trunc(LIMIT((1.0f/law_freq) * sample_rate, 1, max_law_p));
	float increment;
	float lin_int, lin_inc;
	int track;
	int fph;
	LADSPA_Data out = 0.0f;
	int i;
	int dl_used = (delay_depth * sample_rate) / 1000;
	float inc_base = 1000.0f * (float)BASE_BUFFER;
	
	delay_depth *= 2.0f;
	
	for (pos = 0; pos < sample_count; pos++) {
	        // Write into the delay line
	        delay_line[delay_pos] = input[pos];
	        z0 = delay_line[MOD(delay_pos - dl_used, delay_line_length)] + 0.12919609397f*z1 - 0.31050847f*z2;
	        out = sat(z0*0.20466966f + z1*0.40933933f + z2*0.40933933f,
	                        -0.23f, 6.3f);
	        z2 = z1; z1 = z0;
	        delay_pos = (delay_pos + 1) % delay_line_length;
	
	        // Once per law 'frequency' splurge a new set of windowed data
	        // into the law buffer.
	        if ((count++ % law_p) == 0) {
	                // Value for amplitude of law peak
	                float n = (float)rand() / (float)RAND_MAX;
	
	                // Is the window table up to date?
	                if (last_law_p != law_p) {
	                        rc_window(window, meta_window, law_p, max_law_p);
	                        last_law_p = law_p;
	                }
	
	                // Interpolate into law tables
	                for (i = 0; i<law_p*2; i++) {
	                        law_tbl[(i+law_pos) % (max_law_p*2)] += window[i] * n;
	                }
	        }
	
	        increment = inc_base / (delay_depth * law_tbl[law_pos] + 0.2);
	        fph = f_trunc(phase);
	        last_phase = fph;
	        lin_int = phase - (float)fph;
	        out -= sat(LIN_INTERP(lin_int, buffer[(fph+1) % buffer_size],
	         buffer[(fph+2) % buffer_size]), -0.3f, 5.7f);
	        phase += increment;
	        lin_inc = 1.0f / (floor(phase) - last_phase + 1);
	        lin_inc = lin_inc > 1.0f ? 1.0f : lin_inc;
	        lin_int = 0.0f;
	        for (track = last_phase; track < phase; track++) {
	                lin_int += lin_inc;
	                buffer[track % buffer_size] =
	                 LIN_INTERP(lin_int, last_in, input[pos]);
	        }
	        last_in = input[pos];
	        buffer_write(output[pos], out);
	        if (phase >= buffer_size) {
	                phase -= buffer_size;
	        }
	
	        // Zero the used law buffer
	        law_tbl[law_pos] = 0.0f;
	
	        // Increment law pos
	        law_pos = (law_pos + 1) % (max_law_p * 2);
	}
	
	// Store current phase in instance
	plugin_data->phase = phase;
	plugin_data->last_phase = last_phase;
	plugin_data->last_in = last_in;
	plugin_data->count = count;
	plugin_data->law_pos = law_pos;
	plugin_data->last_law_p = last_law_p;
	plugin_data->delay_pos = delay_pos;
	plugin_data->z0 = z0;
	plugin_data->z1 = z1;
	plugin_data->z2 = z2;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainRetroFlange(LADSPA_Handle instance, LADSPA_Data gain) {
	((RetroFlange *)instance)->run_adding_gain = gain;
}

void runAddingRetroFlange(LADSPA_Handle instance, unsigned long sample_count) {
	RetroFlange *plugin_data = (RetroFlange *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Average stall (ms) (float value) */
	LADSPA_Data delay_depth = *(plugin_data->delay_depth);

	/* Flange frequency (Hz) (float value) */
	LADSPA_Data law_freq = *(plugin_data->law_freq);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data * buffer = plugin_data->buffer;
	long buffer_size = plugin_data->buffer_size;
	long count = plugin_data->count;
	LADSPA_Data * delay_line = plugin_data->delay_line;
	int delay_line_length = plugin_data->delay_line_length;
	int delay_pos = plugin_data->delay_pos;
	LADSPA_Data last_in = plugin_data->last_in;
	int last_law_p = plugin_data->last_law_p;
	int last_phase = plugin_data->last_phase;
	int law_pos = plugin_data->law_pos;
	float * law_tbl = plugin_data->law_tbl;
	int max_law_p = plugin_data->max_law_p;
	float * meta_window = plugin_data->meta_window;
	float phase = plugin_data->phase;
	long sample_rate = plugin_data->sample_rate;
	float * window = plugin_data->window;
	LADSPA_Data z0 = plugin_data->z0;
	LADSPA_Data z1 = plugin_data->z1;
	LADSPA_Data z2 = plugin_data->z2;

	long int pos;
	int law_p = f_trunc(LIMIT((1.0f/law_freq) * sample_rate, 1, max_law_p));
	float increment;
	float lin_int, lin_inc;
	int track;
	int fph;
	LADSPA_Data out = 0.0f;
	int i;
	int dl_used = (delay_depth * sample_rate) / 1000;
	float inc_base = 1000.0f * (float)BASE_BUFFER;
	
	delay_depth *= 2.0f;
	
	for (pos = 0; pos < sample_count; pos++) {
	        // Write into the delay line
	        delay_line[delay_pos] = input[pos];
	        z0 = delay_line[MOD(delay_pos - dl_used, delay_line_length)] + 0.12919609397f*z1 - 0.31050847f*z2;
	        out = sat(z0*0.20466966f + z1*0.40933933f + z2*0.40933933f,
	                        -0.23f, 6.3f);
	        z2 = z1; z1 = z0;
	        delay_pos = (delay_pos + 1) % delay_line_length;
	
	        // Once per law 'frequency' splurge a new set of windowed data
	        // into the law buffer.
	        if ((count++ % law_p) == 0) {
	                // Value for amplitude of law peak
	                float n = (float)rand() / (float)RAND_MAX;
	
	                // Is the window table up to date?
	                if (last_law_p != law_p) {
	                        rc_window(window, meta_window, law_p, max_law_p);
	                        last_law_p = law_p;
	                }
	
	                // Interpolate into law tables
	                for (i = 0; i<law_p*2; i++) {
	                        law_tbl[(i+law_pos) % (max_law_p*2)] += window[i] * n;
	                }
	        }
	
	        increment = inc_base / (delay_depth * law_tbl[law_pos] + 0.2);
	        fph = f_trunc(phase);
	        last_phase = fph;
	        lin_int = phase - (float)fph;
	        out -= sat(LIN_INTERP(lin_int, buffer[(fph+1) % buffer_size],
	         buffer[(fph+2) % buffer_size]), -0.3f, 5.7f);
	        phase += increment;
	        lin_inc = 1.0f / (floor(phase) - last_phase + 1);
	        lin_inc = lin_inc > 1.0f ? 1.0f : lin_inc;
	        lin_int = 0.0f;
	        for (track = last_phase; track < phase; track++) {
	                lin_int += lin_inc;
	                buffer[track % buffer_size] =
	                 LIN_INTERP(lin_int, last_in, input[pos]);
	        }
	        last_in = input[pos];
	        buffer_write(output[pos], out);
	        if (phase >= buffer_size) {
	                phase -= buffer_size;
	        }
	
	        // Zero the used law buffer
	        law_tbl[law_pos] = 0.0f;
	
	        // Increment law pos
	        law_pos = (law_pos + 1) % (max_law_p * 2);
	}
	
	// Store current phase in instance
	plugin_data->phase = phase;
	plugin_data->last_phase = last_phase;
	plugin_data->last_in = last_in;
	plugin_data->count = count;
	plugin_data->law_pos = law_pos;
	plugin_data->last_law_p = last_law_p;
	plugin_data->delay_pos = delay_pos;
	plugin_data->z0 = z0;
	plugin_data->z1 = z1;
	plugin_data->z2 = z2;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	retroFlangeDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (retroFlangeDescriptor) {
		retroFlangeDescriptor->UniqueID = 1208;
		retroFlangeDescriptor->Label = strdup("retroFlange");
		retroFlangeDescriptor->Properties =
		 0;
		retroFlangeDescriptor->Name =
		 strdup("Retro Flanger");
		retroFlangeDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		retroFlangeDescriptor->Copyright =
		 strdup("GPL");
		retroFlangeDescriptor->PortCount = 4;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(4,
		 sizeof(LADSPA_PortDescriptor));
		retroFlangeDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(4,
		 sizeof(LADSPA_PortRangeHint));
		retroFlangeDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(4, sizeof(char*));
		retroFlangeDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Average stall (ms) */
		port_descriptors[RETROFLANGE_DELAY_DEPTH] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[RETROFLANGE_DELAY_DEPTH] =
		 strdup("Average stall (ms)");
		port_range_hints[RETROFLANGE_DELAY_DEPTH].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[RETROFLANGE_DELAY_DEPTH].LowerBound = 0;
		port_range_hints[RETROFLANGE_DELAY_DEPTH].UpperBound = 10;

		/* Parameters for Flange frequency (Hz) */
		port_descriptors[RETROFLANGE_LAW_FREQ] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[RETROFLANGE_LAW_FREQ] =
		 strdup("Flange frequency (Hz)");
		port_range_hints[RETROFLANGE_LAW_FREQ].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[RETROFLANGE_LAW_FREQ].LowerBound = 0.5;
		port_range_hints[RETROFLANGE_LAW_FREQ].UpperBound = 8;

		/* Parameters for Input */
		port_descriptors[RETROFLANGE_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[RETROFLANGE_INPUT] =
		 strdup("Input");
		port_range_hints[RETROFLANGE_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[RETROFLANGE_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[RETROFLANGE_OUTPUT] =
		 strdup("Output");
		port_range_hints[RETROFLANGE_OUTPUT].HintDescriptor = 0;

		retroFlangeDescriptor->activate = NULL;
		retroFlangeDescriptor->cleanup = cleanupRetroFlange;
		retroFlangeDescriptor->connect_port = connectPortRetroFlange;
		retroFlangeDescriptor->deactivate = NULL;
		retroFlangeDescriptor->instantiate = instantiateRetroFlange;
		retroFlangeDescriptor->run = runRetroFlange;
		retroFlangeDescriptor->run_adding = runAddingRetroFlange;
		retroFlangeDescriptor->set_run_adding_gain = setRunAddingGainRetroFlange;
	}
}

void _fini() {
	int i;

	if (retroFlangeDescriptor) {
		free((char *)retroFlangeDescriptor->Label);
		free((char *)retroFlangeDescriptor->Name);
		free((char *)retroFlangeDescriptor->Maker);
		free((char *)retroFlangeDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)retroFlangeDescriptor->PortDescriptors);
		for (i = 0; i < retroFlangeDescriptor->PortCount; i++)
			free((char *)(retroFlangeDescriptor->PortNames[i]));
		free((char **)retroFlangeDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)retroFlangeDescriptor->PortRangeHints);
		free(retroFlangeDescriptor);
	}

}
