#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#define VALVE_Q                        0
#define VALVE_DIST                     1
#define VALVE_INPUT                    2
#define VALVE_OUTPUT                   3

LADSPA_Descriptor *valveDescriptor = NULL;

typedef struct {
	LADSPA_Data *q;
	LADSPA_Data *dist;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data  itm1;
	LADSPA_Data  otm1;
	LADSPA_Data run_adding_gain;
} Valve;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return valveDescriptor;
	default:
		return NULL;
	}
}

void activateValve(LADSPA_Handle instance) {
	Valve *plugin_data = (Valve *)instance;
	LADSPA_Data itm1 = plugin_data->itm1;
	LADSPA_Data otm1 = plugin_data->otm1;
	itm1 = 0.0f;
	otm1 = 0.0f;
	plugin_data->itm1 = itm1;
	plugin_data->otm1 = otm1;

}

void cleanupValve(LADSPA_Handle instance) {
	free(instance);
}

void connectPortValve(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Valve *plugin;

	plugin = (Valve *)instance;
	switch (port) {
	case VALVE_Q:
		plugin->q = data;
		break;
	case VALVE_DIST:
		plugin->dist = data;
		break;
	case VALVE_INPUT:
		plugin->input = data;
		break;
	case VALVE_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateValve(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Valve *plugin_data = (Valve *)malloc(sizeof(Valve));
	plugin_data->run_adding_gain = 1.0f;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runValve(LADSPA_Handle instance, unsigned long sample_count) {
	Valve *plugin_data = (Valve *)instance;

	/* Distortion level (float value) */
	LADSPA_Data q = *(plugin_data->q);

	/* Distortion character (float value) */
	LADSPA_Data dist = *(plugin_data->dist);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data itm1 = plugin_data->itm1;
	LADSPA_Data otm1 = plugin_data->otm1;

	unsigned long pos;
	LADSPA_Data fx;
	
	q = q - 1.0f;
	dist = dist * 40.0f + 0.1f;
	
	if (q == 0.0f) {
	        for (pos = 0; pos < sample_count; pos++) {
	                if (input[pos] == q) {
	                        fx = 1.0f / dist;
	                } else {
	                        fx = input[pos] / (1.0f - exp(-dist * input[pos]));
	                }
	                otm1 = 0.999f * otm1 + fx - itm1;
	                itm1 = fx;
	                buffer_write(output[pos], otm1);
	        }
	} else {
	        for (pos = 0; pos < sample_count; pos++) {
	                if (input[pos] == q) {
	                        fx = 1.0f / dist + q / (1.0f - exp(dist * q));
	                } else {
	                        fx = (input[pos] - q) /
	                         (1.0f - exp(-dist * (input[pos] - q))) +
	                         q / (1.0f - exp(dist * q));
	                }
	                otm1 = 0.999f * otm1 + fx - itm1;
	                itm1 = fx;
	                buffer_write(output[pos], otm1);
	        }
	}
	
	plugin_data->itm1 = itm1;
	plugin_data->otm1 = otm1;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainValve(LADSPA_Handle instance, LADSPA_Data gain) {
	((Valve *)instance)->run_adding_gain = gain;
}

void runAddingValve(LADSPA_Handle instance, unsigned long sample_count) {
	Valve *plugin_data = (Valve *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Distortion level (float value) */
	LADSPA_Data q = *(plugin_data->q);

	/* Distortion character (float value) */
	LADSPA_Data dist = *(plugin_data->dist);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;
	LADSPA_Data itm1 = plugin_data->itm1;
	LADSPA_Data otm1 = plugin_data->otm1;

	unsigned long pos;
	LADSPA_Data fx;
	
	q = q - 1.0f;
	dist = dist * 40.0f + 0.1f;
	
	if (q == 0.0f) {
	        for (pos = 0; pos < sample_count; pos++) {
	                if (input[pos] == q) {
	                        fx = 1.0f / dist;
	                } else {
	                        fx = input[pos] / (1.0f - exp(-dist * input[pos]));
	                }
	                otm1 = 0.999f * otm1 + fx - itm1;
	                itm1 = fx;
	                buffer_write(output[pos], otm1);
	        }
	} else {
	        for (pos = 0; pos < sample_count; pos++) {
	                if (input[pos] == q) {
	                        fx = 1.0f / dist + q / (1.0f - exp(dist * q));
	                } else {
	                        fx = (input[pos] - q) /
	                         (1.0f - exp(-dist * (input[pos] - q))) +
	                         q / (1.0f - exp(dist * q));
	                }
	                otm1 = 0.999f * otm1 + fx - itm1;
	                itm1 = fx;
	                buffer_write(output[pos], otm1);
	        }
	}
	
	plugin_data->itm1 = itm1;
	plugin_data->otm1 = otm1;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	valveDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (valveDescriptor) {
		valveDescriptor->UniqueID = 1209;
		valveDescriptor->Label = strdup("valve");
		valveDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		valveDescriptor->Name =
		 strdup("Valve saturation");
		valveDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		valveDescriptor->Copyright =
		 strdup("GPL");
		valveDescriptor->PortCount = 4;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(4,
		 sizeof(LADSPA_PortDescriptor));
		valveDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(4,
		 sizeof(LADSPA_PortRangeHint));
		valveDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(4, sizeof(char*));
		valveDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Distortion level */
		port_descriptors[VALVE_Q] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[VALVE_Q] =
		 strdup("Distortion level");
		port_range_hints[VALVE_Q].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[VALVE_Q].LowerBound = 0;
		port_range_hints[VALVE_Q].UpperBound = 1;

		/* Parameters for Distortion character */
		port_descriptors[VALVE_DIST] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[VALVE_DIST] =
		 strdup("Distortion character");
		port_range_hints[VALVE_DIST].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[VALVE_DIST].LowerBound = 0;
		port_range_hints[VALVE_DIST].UpperBound = 1;

		/* Parameters for Input */
		port_descriptors[VALVE_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[VALVE_INPUT] =
		 strdup("Input");
		port_range_hints[VALVE_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[VALVE_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[VALVE_OUTPUT] =
		 strdup("Output");
		port_range_hints[VALVE_OUTPUT].HintDescriptor = 0;

		valveDescriptor->activate = activateValve;
		valveDescriptor->cleanup = cleanupValve;
		valveDescriptor->connect_port = connectPortValve;
		valveDescriptor->deactivate = NULL;
		valveDescriptor->instantiate = instantiateValve;
		valveDescriptor->run = runValve;
		valveDescriptor->run_adding = runAddingValve;
		valveDescriptor->set_run_adding_gain = setRunAddingGainValve;
	}
}

void _fini() {
	int i;

	if (valveDescriptor) {
		free((char *)valveDescriptor->Label);
		free((char *)valveDescriptor->Name);
		free((char *)valveDescriptor->Maker);
		free((char *)valveDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)valveDescriptor->PortDescriptors);
		for (i = 0; i < valveDescriptor->PortCount; i++)
			free((char *)(valveDescriptor->PortNames[i]));
		free((char **)valveDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)valveDescriptor->PortRangeHints);
		free(valveDescriptor);
	}

}
