// Copyright (C) 2009-2009 Martin Sandve Alnæs
// Licensed under the GNU LGPL Version 3.0.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves a hyperelasticity problem
// in 3D using the Saint-Vernant-Kirchoff model.

#include <iostream>
#include <vector>
#include <dolfin.h>
#include "generated_code/HyperElasticitySVK.h"

using dolfin::message;
using dolfin::Function;
using dolfin::Constant;
using dolfin::MeshFunction;
using dolfin::FunctionSpace;
using dolfin::SubDomain;
using dolfin::uint;
using dolfin::UnitCube;
using dolfin::DirichletBC;
using dolfin::BoundaryCondition;
using dolfin::VariationalProblem;
using dolfin::Vector;
using dolfin::Matrix;
using dolfin::File;

using namespace HyperElasticitySVK;

using std::cout;
using std::endl;

// ---------------------------------------- Functions

class BodyForce: public Function
{
public:
  BodyForce(const FunctionSpace & V): Function(V) {}

  void eval(double* values, const double* x) const
  {
    const double dx = x[0] - 0.5;
    const double dy = x[1] - 0.5;
    const double dz = x[2] - 0.5;

    values[0] = 0.0;
    values[1] = 0.0;
    values[2] = 0.0;
  }
};

class BoundaryPressure: public Function
{
public:
  BoundaryPressure(const FunctionSpace & V): Function(V) {}

  void eval(double* values, const double* x) const
  {
    values[0] = 0.0;
  }
};

// Dirichlet boundary condition for rotation at right end
class Rotation : public Function
{
  void eval(double* values, const double* x) const
  {
    // Center of rotation
    double y0 = 0.5;
    double z0 = 0.5;
    
    // Angle of rotation (60 degrees)
    double theta = 0.5236*2;
    
    // New coordinates
    double y = y0 + (x[1] - y0)*cos(theta) - (x[2] - z0)*sin(theta);
    double z = z0 + (x[1] - y0)*sin(theta) + (x[2] - z0)*cos(theta);
    
    // Clamp at right end
    values[0] = 0.0;
    values[1] = y - x[1];
    values[2] = z - x[2];
  }
};

// ---------------------------------------- Subdomains

// Sub domain for clamp at left end
class Left : public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    return on_boundary && x[0] < 0.0 + DOLFIN_EPS;
  }
};

// Sub domain for rotation at right end
class Right : public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    return on_boundary && x[0] > 1.0 - DOLFIN_EPS;
  }
};

class EntireBoundary: public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    return on_boundary;
  }
};

// ---------------------------------------- Main program

int main(int argc, char**argv)
{
    // Geometry
    message("Mesh");
    unsigned n = 12;
    UnitCube mesh(n, n, n);

    // Boundaries
    message("Boundaries");
    unsigned pressure_boundary_marker = 0;
    unsigned left_boundary_marker     = 1;
    unsigned right_boundary_marker    = 2;

    MeshFunction<unsigned> boundaries(mesh, mesh.geometry().dim()-1);

    EntireBoundary entire_boundary;
    entire_boundary.mark(boundaries, pressure_boundary_marker);
    
    Left left_boundary;
    left_boundary.mark(boundaries, left_boundary_marker);
    
    Right right_boundary;
    right_boundary.mark(boundaries, right_boundary_marker);

    // Function spaces
    message("Function spaces");
    Form_a_F::TestSpace V(mesh);
    CoefficientSpace_sigma_0 P(mesh);
    CoefficientSpace_mu C(mesh);
    
    // Coefficient functions
    message("Functions");
    
    // Displacement at current and two previous timesteps
    Function u(V);
    u.vector().zero();
    
    // External forces
    BodyForce g(V);
    BoundaryPressure sigma_0(P);

    // BCs
    Constant uleft(3, 0.0);
    Rotation uright;

    // Material parameters
    Constant mu(4.0);
    Constant lamda(6.0);

    // Collect coefficients
    CoefficientSet coeffs;
    coeffs.u       = u;
    coeffs.g       = g;
    coeffs.sigma_0 = sigma_0;
    coeffs.mu      = mu;
    coeffs.lamda   = lamda;

    coeffs.disp();    

    // Forms
    message("Forms");
    Form_a_f f(coeffs);
    Form_a_F L(V, coeffs);
    Form_a_J a(V, V, coeffs);

    // Setup boundary conditions
    message("Boundary conditions");

    // Dirichlet boundary conditions
    DirichletBC leftbc(V, uleft, boundaries, left_boundary_marker);
    DirichletBC rightbc(V, uright, boundaries, right_boundary_marker);
    
    std::vector<BoundaryCondition*> bcs;
    bcs.push_back(&leftbc);
    bcs.push_back(&rightbc);

    // Create variational problem (a, L, bcs, cell_domains, exterior_facet_domains, interior_facet_domains, nonlinear)
    message("Variational problem");
    VariationalProblem problem(a, L, bcs, 0, &boundaries, 0, true);
    
    // Solve!
    message("Solving");
    problem.set("linear solver", "direct");
    //problem.set("linear solver", "iterative");
    problem.solve(u);

    // Postprocessing
    message("Postprocessing");
    
    // Compute internal energy for u and exact solution
    double f_value = assemble(f);

    // Compute F at u for verification
    Vector F;
    problem.F(F, u.vector());

    // Print limits of u
    cout << endl;
    cout << "==========================" << endl;
    cout << "Norm of u_h = " << u.vector().norm() << endl;
    cout << "Min  of u_h = " << u.vector().min() << endl;
    cout << "Max  of u_h = " << u.vector().max() << endl;
    cout << "==========================" << endl;

    cout << endl;
    cout << "==========================" << endl;
    cout << "f(u) = " << f_value << endl;
    cout << "Norm of F = " << F.norm() << endl;
    cout << "Min  of F = " << F.min() << endl;
    cout << "Max  of F = " << F.max() << endl;
    cout << "==========================" << endl;
    
    // Write displacement to file
    message("Writing to file");
    
    File ufile("u.pvd");
    ufile << u;

    //plot(w);
    message("Done!");
    
}

