#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
This module contains the Just-In-Time compiler tools.
The main function is jit(form, options), see its documentation.
"""

__author__ = "Martin Sandve Alnes"
__date__   = "2008-09-04 -- 2009-04-23"
__copyright__ = "(C) 2008-2009 Martin Sandve Alnes and Simula Resarch Laboratory"
__license__  = "GNU GPL Version 2, or (at your option) any later version"


import shutil
import os
import hashlib
import instant
import ufc_utils
import ufl
import ufl.algorithms

from sfc.common import version
from sfc.common import default_options
from sfc.common import ParameterDict
from sfc.common import get_abs_ufc_h_path
from sfc.common import sfc_debug, sfc_info, sfc_warning, sfc_error, sfc_assert, write_file
from sfc.common.names import short_name, form_classname, dof_map_classname, finite_element_classname, base_element_classname
from sfc.codegeneration import generate_code, compiler_input

def cache_signature(input, options):
    "A signature string uniquely identifying jit input."
    if not isinstance(input, list):
        input = [input]
    newinput = []
    for i in input:
        if isinstance(i, ufl.Form):
            i = i.form_data()
        if isinstance(i, ufl.algorithms.FormData):
            i = i.form
        newinput.append(i)
    return "sfc.jit(%r, %r) # SFC version %r" % (newinput, options, version)

def compile_module(module_dir, hfilenames, cfilenames, signature, options):
    """Assuming an existing directory module_dir with source files
    hfilenames, cfilenames, create a python extension module and compile it."""

    orig_dir = os.getcwd()
    try:
        # headers to #include in swig file
        system_headers = ["iostream", "stdexcept", "cmath", "ufc.h", "tr1/memory"]

        cppargs = options.compilation.cppargs
        if options.compilation.enable_debug_code:
            cppargs.append("-DSFCDEBUG")

        module_name = module_dir if signature is None else None

        module = ufc_utils.build_ufc_module(h_files = hfilenames,
                     source_directory   = module_dir,
                     sources            = cfilenames,
                     system_headers     = system_headers,
                     cppargs            = cppargs,
                     signature          = signature,
                     cache_dir          = options.compilation.cache_dir,
                     modulename         = module_name,
                     generate_interface = options.compilation.generate_interface,
                     generate_setup     = options.compilation.generate_setup)
        sfc_info("Successfully compiled module in '%s'." % module_dir)
        return module
        # ...
    finally:
        os.chdir(orig_dir)

def jit(input, options = None):
    """Generate code from input and options.
    
    @param input:
        TODO
    @param options:
        TODO
    @return:
        FIXME: Agree on return values with FFC! Currently FFC returns (ufc_form, module, formdata), and dolfin assumes this
    """

    # To handle all input uniformly
    list_input = isinstance(input, list)
    ufl_elements, formdatas = compiler_input(input)
    
    # Merge options with defaults
    if options is None:
        options = default_options()
    if isinstance(options, dict) and not isinstance(options, ParameterDict):
        o = options
        options = default_options()
        options.update(o)
    
    # Default prefix shows where the module was compiled
    prefix = options.code.prefix
    if not prefix:
        prefix = "sfc_jit"

    # Build module signature and classnames from input
    signature = cache_signature(input, options)
    formclassnames = [form_classname(fd.form, options) for fd in formdatas]
    dmclassnames = [dof_map_classname(e) for e in ufl_elements]
    feclassnames = [finite_element_classname(e) for e in ufl_elements]

    # Placeholder for the up and coming compiled module
    module = None

    # Hacky hook for profiling without the C++ compilation
    if options.compilation.skip:
        pass
    else:
        # Possibly force recompilation
        if options.compilation.overwrite_cache:
            sfc_warning("Complete implementation of overwrite_cache not done, but skipping cache lookup before code generation.")
            # TODO: delete existing module with signature or module_name
        else:
            # Check cache before generating code
            module = instant.import_module(signature)
            # Construct ufc::form object
            if module:
                sfc_info("Found module in cache.")
    
    if module is None:
        # Store original path, to restore later
        orig_dir = os.getcwd()

        # Build module name, then shorten if it's too long
        module_dir_name = "_".join(["_".join(formclassnames), "_".join(dmclassnames), "_".join(feclassnames)])
        module_dir_name = short_name(prefix, module_dir_name)
        
        # Make sure the module directory is there and empty
        module_dir = os.path.abspath(module_dir_name)
        if os.path.exists(module_dir):
            sfc_warning("Possibly overwriting files in existing module directory '%s'." % module_dir)
        else:
            os.mkdir(module_dir)
        
        # Enter module directory and generate code!
        try:
            os.chdir(module_dir)
            
            # Write options, signature and form representations to files beside code
            write_file("options.repr", repr(options))
            write_file("signature", signature)
            for fd in formdatas:
                write_file("%s.repr" % short_name("form_", fd.name), repr(fd.form))
            for ue in ufl_elements:
                write_file("%s.repr" % short_name("element_", base_element_classname(ue)), repr(ue))
            
            # Generate code!
            hfilenames, cfilenames = generate_code(ufl_elements + formdatas, options)
            
            # Hook for profiling without the C++ compilation
            if options.compilation.skip:
                return None

        finally:
            # Restore original path
            os.chdir(orig_dir)
        
        # Invoke Instant to compile generated files as a Python extension module
        module = compile_module(module_dir, hfilenames, cfilenames, signature, options)
        
        # Remove temporary module directory if code was copied to cache
        placed_in_cache = False # TODO
        if placed_in_cache:
            sfc_info("Module was placed in cache, deleting module directory '%s'." % module_dir)
            shutil.rmtree(module_dir)
        
        if module is None:
            sfc_error("Failed to load compiled module from cache.")
    
    # If we get here, 'module' should be our compiled module.
    
    # TODO: Handle mixed list of forms and elements?
    
    # Got forms, return forms
    if formdatas:
        ufc_forms = [getattr(module, name)() for name in formclassnames]
        if list_input:
            return ufc_forms, module, formdatas
        else:
            return ufc_forms[0], module, formdatas[0]

    # Got elements, return elements
    elif ufl_elements:
        ufc_elements = [(getattr(module, fe)(), getattr(module, dm)()) for (fe, dm) in zip(feclassnames, dmclassnames)]
        if list_input:
            return ufc_elements
        else:
            return ufc_elements[0]
    
    msg = "Nothing to return, this shouldn't happen.\ninput =\n%r" % repr(input)
    sfc_error(msg)

