/* nmcacheentry.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2001-2008 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "nmcacheentry.hh"
#include "dirfiltersettings.hh"
#include "dirsortsettings.hh"
#include "dirbookmarkssettings.hh"
#include "wconfig.h"
#include <algorithm>

NMCacheEntry::NMCacheEntry( std::auto_ptr<Verzeichnis> verz,
                            DirFilterSettings *dir_filter_sets,
                            DirSortSettings *dir_sort_sets,
                            DirBookmarksSettings *dir_bookmarks_sets ) : _active_is_visible( false ),
                                                                         _last_active_row( -1 ),
                                                                         _verz( verz ),
                                                                         _dir_filter_sets( dir_filter_sets ),
                                                                         _dir_sort_sets( dir_sort_sets ),
                                                                         m_dir_bookmarks_sets( dir_bookmarks_sets ),
                                                                         _filter_serial_nr( 0 ),
                                                                         _sort_serial_nr( 0 ),
                                                                         m_bookmarks_serial_nr( 0 )
{
  int i;

  _pos = 0;
  _xpos = 0;

  for ( i = 0; i < 2; i++ ) {
    _files[i] = -1;
    _dirs[i] = -1;
  }
  for ( i = 0; i < 2; i++ ) {
    _files_s[i] = -1;
    _dirs_s[i] = -1;
  }

  _activefe = NULL;
  _checkfe = NULL;
  _firstnullft = NULL;
  
  checkDirSettings( true );
  
  _dontcheck = false;
}

NMCacheEntry::~NMCacheEntry()
{
}

void NMCacheEntry::subEntryFromStats( const FileEntry *fe )
{
    if ( getVerz() == NULL )
        return;
    if ( fe == NULL )
        return;
    
    if ( fe->isDir() == true ) {
        if ( strcmp( fe->name, ".." ) != 0 ) {
            _dirs[0]--;
            if ( fe->dirsize >= 0 )
                _dirs_s[0] -= fe->dirsize;
            else
                _dirs_s[0] -= fe->size();
            if ( fe->select == true ) {
                _dirs[1]--;
                if ( fe->dirsize >= 0 )
                    _dirs_s[1] -= fe->dirsize;
                else
                    _dirs_s[1] -= fe->size();
            }
        }
    } else {
        _files[0]--;
        _files_s[0] -= fe->size();
        if ( fe->select == true ) {
            _files[1]--;
            _files_s[1] -= fe->size();
        }
    }
}

void NMCacheEntry::updateStatEntrySelected( const FileEntry *fe )
{
    if ( getVerz() == NULL )
        return;
    if ( fe == NULL )
        return;
    
    if ( fe->isDir() == true ) {
        _dirs[1]++;
        if ( fe->dirsize >= 0 )
            _dirs_s[1] += fe->dirsize;
        else
            _dirs_s[1] += fe->size();
    } else {
        _files[1]++;
        _files_s[1] += fe->size();
    }
}

void NMCacheEntry::updateStatEntryDeselected( const FileEntry *fe )
{
    if ( getVerz() == NULL )
        return;
    if ( fe == NULL )
        return;
    
    if ( fe->isDir() == true ) {
        _dirs[1]--;
        if ( fe->dirsize >= 0 )
            _dirs_s[1] -= fe->dirsize;
        else
            _dirs_s[1] -= fe->size();
    } else {
        _files[1]--;
        _files_s[1] -= fe->size();
    }
}

/*
 * NMCacheEntry::recalcStats
 *
 * recalculate the nr of bytes/files...
 * resets also activefe when it's not visible
 */
void NMCacheEntry::recalcStats()
{
  if ( getVerz() == NULL )
      return;

  int i;

  for ( i = 0; i < 2; i++ ) {
    _files[i] = 0;
    _dirs[i] = 0;
  }

  for ( i = 0; i < 2; i++ ) {
    _files_s[i] = 0;
    _dirs_s[i] = 0;
  }

  for ( Verzeichnis::verz_it fe_it1 = begin();
        fe_it1 != end();
        fe_it1++ ) {
      FileEntry *fe = *fe_it1;
      if ( fe->use == true ) {
          if ( fe->isDir() == true ) {
              if ( strcmp( fe->name, ".." ) != 0 ) {
                  _dirs[0]++;
                  if ( fe->dirsize >= 0 )
                      _dirs_s[0] += fe->dirsize;
                  else
                      _dirs_s[0] += fe->size();
                  if ( fe->select == true ) {
                      _dirs[1]++;
                      if ( fe->dirsize >= 0 )
                          _dirs_s[1] += fe->dirsize;
                      else
                          _dirs_s[1] += fe->size();
                  }
              }
          } else {
              _files[0]++;
              _files_s[0] += fe->size();
              if ( fe->select == true ) {
                  _files[1]++;
                  _files_s[1] += fe->size();
              }
          }
      } else {
          if ( fe == _activefe ) _activefe = NULL;
      }
  }
}

/*
 * finds next visible entry without filetype
 * effect: advance _checkfe
 */
void NMCacheEntry::next_checkfe()
{
    if ( _checkfe == NULL )
        return;
    
    if ( getVerz() != NULL && dirOpened() == true ) {

        Verzeichnis::verz_it fe_it1 = std::find( begin(), end(), _checkfe );

        if ( fe_it1 != end() )
            fe_it1++;

        for ( _checkfe = NULL;
              fe_it1 != end();
              fe_it1++ ) {
            
            _checkfe = *fe_it1;
            
            if ( _checkfe->use == true ) {
                // Pruefen, ob Dateityp noch nicht erkannt wurde
                // TODO: Jetzt reicht noch auf NULL pruefen, spaeter muss auf den NotYetChecked geprueft
                // werden
                if ( _checkfe->filetype == NULL ) {
                    if ( _checkfe->isLink == true &&
                         _checkfe->isCorrupt == false &&
                         _checkfe->isDir() == false )
                        break;
                    if ( _checkfe->isDir() == false )
                        break;
                }
            }
            
            _checkfe = NULL;
        }
    } else {
        _checkfe = NULL;
    }
}

/*
 * reset all filetypes to NULL
 * effect: reset _checkfe and _firstnullft
 */
void NMCacheEntry::restartcheck()
{
    if ( getVerz() != NULL && dirOpened() == true ) {
        for ( Verzeichnis::verz_it fe_it1 = begin();
              fe_it1 != end();
              fe_it1++ ) {
            FileEntry *fe = *fe_it1;
            fe->filetype = NULL;
            fe->clearCustomColor();
        }
        _checkfe = *begin();;
    } else {
        _checkfe = NULL;
    }
    
    next_checkfe();
    _firstnullft = _checkfe;
}

/*
 * resets _checkfe to first null filetype
 * effect: reset _checkfe and _firstnullft
 */
void NMCacheEntry::reset_checkfe()
{
    _checkfe = NULL;
    if ( getVerz() != NULL && dirOpened() == true ) {
        if ( begin() != end() )
            _checkfe = *begin();
    }
    
    next_checkfe();
    _firstnullft = _checkfe;
}

/*
 * resets _firstnullft to first null filetype
 * effect: advance _firstnullft
 */
void NMCacheEntry::reset_firstnullft()
{
    _firstnullft = NULL;
    if ( getVerz() != NULL && dirOpened() == true ) {
        if ( begin() != end() ) {
            _firstnullft = *begin();
            next_firstnullft();
        }
    }
}

/*
 * finds next visible entry without filetype
 * effect: advance _firstnullft
 */
void NMCacheEntry::next_firstnullft()
{
    if ( _firstnullft == NULL ) return;
    
    if ( getVerz() != NULL && dirOpened() == true ) {

        Verzeichnis::verz_it fe_it1 = std::find( begin(), end(), _firstnullft );

        if ( fe_it1 != end() )
            fe_it1++;

        for ( _firstnullft = NULL;
              fe_it1 != end();
              fe_it1++ ) {
            _firstnullft = *fe_it1;
            if ( _firstnullft->use == true ) {
                // Pruefen, ob Dateityp noch nicht erkannt wurde
                // TODO: Jetzt reicht noch auf NULL pruefen, spaeter muss auf den NotYetChecked geprueft
                // werden
                if ( _firstnullft->filetype == NULL ) {
                    if ( _firstnullft->isLink == true &&
                         _firstnullft->isCorrupt == false &&
                         _firstnullft->isDir() == false )
                        break;
                    if ( _firstnullft->isDir() == false )
                        break;
                }
            }
            _firstnullft = NULL;
        }
    } else {
        _firstnullft = NULL;
    }
}

void NMCacheEntry::checkfordcd()
{
  List *dcd=wconfig->getDontCheckDirs();
  char *tstr;
  const char *dirstr;
  int id;

  _dontcheck=false;
  if ( getVerz() != NULL ) {
    dirstr = getVerz()->getDir();
    if(dcd!=NULL) {
      id=dcd->initEnum();
      tstr=(char*)dcd->getFirstElement(id);
      while(tstr!=NULL) {
        if(strncmp(tstr,dirstr,strlen(tstr))==0) {
          _dontcheck=true;
          break;
        }
        tstr=(char*)dcd->getNextElement(id);
      }
      dcd->closeEnum(id);
    }
  }
}

void NMCacheEntry::reset_dirsizes()
{
    if ( getVerz() == NULL )
        return;
    
    for ( Verzeichnis::verz_it fe_it1 = begin();
          fe_it1 != end();
          fe_it1++ ) {
        FileEntry *fe = *fe_it1;
        if ( fe->isDir() == true ) {
            fe->dirsize = -1;
        }
    }
}

void NMCacheEntry::checkDirSettings( bool force )
{
    if ( _verz.get() == NULL )
        return;

    bool dirty = false;

    if ( m_dir_bookmarks_sets->getSerialNr() != m_bookmarks_serial_nr || force == true ) {
        m_dir_bookmarks_sets->check( *( _verz.get() ) );

        m_bookmarks_serial_nr = m_dir_bookmarks_sets->getSerialNr();
        dirty = true;
    }

    if ( _dir_filter_sets->getSerialNr() != _filter_serial_nr || force == true ) {

        Verzeichnis::verz_it fe_it1 = _verz->begin();
        if ( fe_it1 != _verz->end() )
            fe_it1++;  // skip first entry

        for ( ;
              fe_it1 != _verz->end();
              fe_it1++ ) {
            FileEntry *fe = *fe_it1;
            _dir_filter_sets->check( *fe );
            if ( fe == _activefe && fe->use == false )
                _activefe = NULL;
        }
        
        _filter_serial_nr = _dir_filter_sets->getSerialNr();
        dirty = true;
    }

    if ( _dir_sort_sets->getSerialNr() != _sort_serial_nr || force == true ) {
        _dir_sort_sets->check( *( _verz.get() ) );

        _sort_serial_nr = _dir_sort_sets->getSerialNr();
        dirty = true;
    }

    if ( dirty == true ) {
        recalcStats();
        reset_checkfe();
    }
}

void NMCacheEntry::setVerz( std::auto_ptr<Verzeichnis> verz )
{
    _verz = verz;
    
    _activefe = NULL;

    checkDirSettings( true );
}

Verzeichnis *NMCacheEntry::getVerz()
{
    checkDirSettings();
    return _verz.get();
}

const char *NMCacheEntry::getDir()
{
    /* special case: getDir is independent from filter/sort
     * so directly use _verz which is much faster
     */
    if ( _verz.get() == NULL )
        return NULL;
    return _verz->getDir();
}

void NMCacheEntry::forceUpdate()
{
    // force update by decrementing serial number
    _filter_serial_nr--;
    _sort_serial_nr--;
}

void NMCacheEntry::resetCheckFE2FirstNULLFT()
{
    // getVerz() is intentionally not called as this
    // is used when clearing the list of entries to recognize
    // it wouldn't be wrong to call getVerz() but it can reset
    // activefe if a search filter is active and this is not
    // always wanted
    
    _checkfe = _firstnullft;
}

FileEntry *NMCacheEntry::getActiveFE()
{
    //TODO call getVerz()?
    return _activefe;
}

void NMCacheEntry::setActiveFE( FileEntry *fe )
{
    //TODO call getVerz()?
    _activefe = fe;
}

FileEntry *NMCacheEntry::getCheckFE()
{
    if ( getVerz() == NULL )
        return NULL;
    
    return _checkfe;
}

FileEntry *NMCacheEntry::getFirstNULLFT()
{
    if ( getVerz() == NULL )
        return NULL;
    
    return _firstnullft;
}

int NMCacheEntry::getPos() const
{
    return _pos;
}

void NMCacheEntry::setPos( int pos )
{
    _pos = pos;
}

int NMCacheEntry::getXPos() const
{
    return _xpos;
}

void NMCacheEntry::setXPos( int xpos )
{
    _xpos = xpos;
}

void NMCacheEntry::setActiveIsVisible( bool nv )
{
    _active_is_visible = nv;
}

bool NMCacheEntry::getActiveIsVisible() const
{
    return _active_is_visible;
}

void NMCacheEntry::setLastActiveRow( int row )
{
    _last_active_row = row;
}

int NMCacheEntry::getLastActiveRow() const
{
    return _last_active_row;
}

bool NMCacheEntry::getDontCheck() const
{
    return _dontcheck;
}

long NMCacheEntry::getNrOfFiles( int i )
{
    if ( getVerz() == NULL )
        return 0;
    if ( i < 0 || i > 1 )
        return 0;
    
    return _files[i];
}

loff_t NMCacheEntry::getSizeOfFiles( int i )
{
    if ( getVerz() == NULL )
        return 0;
    if ( i < 0 || i > 1 )
        return 0;
    
    return _files_s[i];
}

long NMCacheEntry::getNrOfDirs( int i )
{
    if ( getVerz() == NULL )
        return 0;
    if ( i < 0 || i > 1 )
        return 0;
    
    return _dirs[i];
}

loff_t NMCacheEntry::getSizeOfDirs( int i )
{
    if ( getVerz() == NULL )
        return 0;
    if ( i < 0 || i > 1 )
        return 0;
    
    return _dirs_s[i];
}

FileEntry *NMCacheEntry::getEntryByID( int ID )
{
    if ( getVerz() == NULL )
        return NULL;
    return getVerz()->getEntryByID( ID );
}

void NMCacheEntry::removeEntry( const FileEntry *fe )
{
    bool searchcheckfe = false,
        searchfirstnullft = false;
    
    if ( fe == getCheckFE() )
        searchcheckfe = true;

    if ( fe == getFirstNULLFT() )
        searchfirstnullft = true;

    getVerz()->removeElement( fe );
    
    subEntryFromStats( fe );
    
    if ( searchcheckfe == true )
        reset_checkfe();
    
    if ( searchfirstnullft == true )
        reset_firstnullft();
}

int NMCacheEntry::getSerialOfIDs()
{
    if ( getVerz() == NULL )
        return Verzeichnis::getInvalidSerial();
    return getVerz()->getSerialOfIDs();
}

Verzeichnis::verz_it NMCacheEntry::begin()
{
    return getVerz()->begin();
}

Verzeichnis::verz_it NMCacheEntry::end()
{
    return getVerz()->end();
}

bool NMCacheEntry::dirOpened()
{
    if ( getVerz() == NULL )
        return false;
    return getVerz()->dirOpened();
}
