/*
 *	$COPYRIGHT$
 *
 *	$Id: xmpi_browse.cc,v 1.3 2000/10/29 19:32:02 bbarrett Exp $
 *
 *	Function:	- program browser panel in builder dialog
 */

#include <Xm/Label.h>
#include <Xm/List.h>
#include <Xm/PanedW.h>
#include <Xm/TextF.h>

#include <stdlib.h>
#include <dirent.h>
#include <limits.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "all_list.h"
#include "lam.h"
#include "xmpi.h"
#include "xmpi_misc.h"
#include "getworkdir.h"
#include "xmpi_browse.h"
#include "xmpi_error.h"
#include "xmpi_run.h"

/*
 * local functions
 */
static void browse_dir_cb(Widget, XtPointer, XmAnyCallbackStruct*);

static void browse_select_cb(Widget, XtPointer, XmListCallbackStruct*);

extern "C"{
static int fnamecmp(void*, void*);
}

/*
 * local variables
 */
static Widget list_w = 0;	       /* directory contents list */

static Widget text_w = 0;	       /* current directory text field */

/*
 *	xmpi_browse_build
 *
 *	Function:	- builds file browse panel
 *	Accepts:	- parent widget
 */
void
xmpi_browse_build(Widget parent_w)
{
  Widget mgr_w;

  char *cwd;

  XmString xstr;

  mgr_w = XtVaCreateWidget("browse_mgr",
			   xmPanedWindowWidgetClass, parent_w,
			   XmNsashWidth, 1,
			   XmNsashHeight, 1,
			   XmNseparatorOn, False,
			   XmNleftAttachment, XmATTACH_POSITION,
			   XmNleftPosition, 1,
			   XmNleftOffset, 5,
			   XmNrightAttachment, XmATTACH_POSITION,
			   XmNrightPosition, 2,
			   XmNtopAttachment, XmATTACH_FORM,
			   XmNbottomAttachment, XmATTACH_FORM,
			   NULL);

  xstr = XmStringCreateSimple((char*) "Browse Programs");
  XtVaCreateManagedWidget("banner",
			  xmLabelWidgetClass, mgr_w,
			  XmNlabelString, xstr,
			  NULL);
  XmStringFree(xstr);

  text_w = XtVaCreateManagedWidget("browse_text",
				   xmTextFieldWidgetClass, mgr_w,
				   NULL);
  XtAddCallback(text_w, XmNactivateCallback, 
		(XtCallbackProc) browse_dir_cb, NULL);

  list_w = XmCreateScrolledList(mgr_w, (char*) "browse_list", NULL, 0);
  XtVaSetValues(list_w,
		XmNscrollBarDisplayPolicy, XmSTATIC,
		XmNscrollHorizontal, False,
		XmNselectionPolicy, XmBROWSE_SELECT,
		NULL);
  XtAddCallback(list_w, XmNdefaultActionCallback,
		(XtCallbackProc) browse_select_cb, NULL);
  XtAddCallback(list_w, XmNbrowseSelectionCallback,
		(XtCallbackProc) browse_select_cb, NULL);
  XtManageChild(list_w);
/*
 * Initialize the text field to the current directory.
 * Fill the list with the current directory contents.
 */
  if ((cwd = getworkdir())) {
    XmTextFieldSetString(text_w, cwd);
    XmTextFieldSetInsertionPosition(text_w, strlen(cwd));
    xmpi_run_set_prog((char*) "", cwd);
    free(cwd);
    xmpi_browse_fill(list_w);
  }
  xmpi_nosash(mgr_w);
  XtManageChild(mgr_w);
}

/*
 *	browse_dir_cb
 *
 *	Function:	- take action on the selected directory
 *	Accepts:	- widget
 *			- client data
 *			- callback ptr
 */
static void
browse_dir_cb(Widget w, XtPointer, 
	      XmAnyCallbackStruct*)
{
  char *dirpath;

  dirpath = XmTextFieldGetString(w);

  if ((dirpath == 0) || (*dirpath == '\0')) {
    XtFree(dirpath);
    return;
  }
/*
 * Change to new directory.
 */
  if (chdir(dirpath)) {
    xmpi_error(w, dirpath);
    XtFree(dirpath);
    return;
  }
  xmpi_run_set_prog((char*) "", dirpath);
  xmpi_browse_fill(list_w);
  XtFree(dirpath);
}

/*
 *	browse_select_cb
 *
 *	Function:	- selects a program or new directory
 *	Accepts:	- widget
 *			- client data
 *			- callback ptr
 */
static void
browse_select_cb(Widget w, XtPointer, 
		 XmListCallbackStruct *cbs)
{
  char *filename;
  char *p;
  struct stat fileinfo;

  XmStringGetLtoR(cbs->item, XmSTRING_DEFAULT_CHARSET, &filename);

  if (lstat(filename, &fileinfo)) {
    xmpi_error(w, filename);
    XtFree(filename);
    return;
  }
/*
 * If the file is a sym-link, stat() the real file.
 */
  if (S_ISLNK(fileinfo.st_mode)) {
    if (stat(filename, &fileinfo)) {
      xmpi_error(w, filename);
      XtFree(filename);
      return;
    }
  }
/*
 * If the file is a directory, change to it.
 */
  if (S_ISDIR(fileinfo.st_mode) &&
      (cbs->reason == (int) XmCR_DEFAULT_ACTION)) {

    if (chdir(filename)) {
      xmpi_error(w, filename);
      XtFree(filename);
      return;
    }
    if ((p = getworkdir()) == 0) {
      xmpi_error(w, (char*) "Cannot get current directory");
      XtFree(filename);
      return;
    }
    XmTextFieldSetString(text_w, p);
    XmTextFieldSetInsertionPosition(text_w, strlen(p));
    xmpi_run_set_prog((char*) "", p);
    xmpi_browse_fill(list_w);
    free(p);
  }
/*
 * Write the selected file into the program text area.
 */
  else if (!S_ISDIR(fileinfo.st_mode)) {

    if ((p = getworkdir()) == 0) {
      xmpi_error(w, (char*) "Cannot get current directory");
      XtFree(filename);
      return;
    }
    xmpi_run_set_prog(filename, p);
    free(p);
  }
  XtFree(filename);
}

/*
 *	xmpi_browse_fill
 *
 *	Function:	- fills a list with the contents of the
 *			  current directory
 *	Accepts:	- list widget
 */
void
xmpi_browse_fill(Widget w)
{
  DIR *pdir;			       /* opened directory */

  struct dirent *pfile;		       /* directory entry ptr */

  XmString xstr;		       /* Motif string */

  struct stat fileinfo;		       /* file status info */

  LIST *sortlist;		       /* list of sorted filenames */

  int fnamelen;			       /* filename length */

  char *fname;			       /* filename */

  char **p;			       /* favourite pointer */

/*
 * Delete all current list entries.
 */
  XmListDeselectAllItems(w);
  XmListDeleteAllItems(w);
/*
 * Open the directory.
 */
  if ((pdir = opendir(".")) == 0)
    xmpi_fail((char*) "xmpi (opendir)");
/*
 * Initialize the list used for sorting.
 */
  sortlist = al_init((int4) sizeof(char *), fnamecmp);

  if (sortlist == 0)
    xmpi_fail((char*) "xmpi (al_init)");
/*
 * Fill the list with the directory's files.
 */
  for (pfile = readdir(pdir); pfile; pfile = readdir(pdir)) {
/*
 * Insert the file in the sorted list.
 * Add a path delimiter if the file is a directory.
 */
    if (lstat(pfile->d_name, &fileinfo)) {
      xmpi_error(w, pfile->d_name);
      continue;
    }
/*
 * If a sym-link, stat() the real file.
 * Ignore any error and accept the link anyway.
 */
    if (S_ISLNK(fileinfo.st_mode)) {
      stat(pfile->d_name, &fileinfo);
    }
    fnamelen = strlen(pfile->d_name);
    if (S_ISDIR(fileinfo.st_mode)) {
      ++fnamelen;
    }
    fname = (char*) malloc((unsigned) fnamelen + 1);

    if (fname == 0)
      xmpi_fail((char*) "xmpi (malloc)");

    strcpy(fname, pfile->d_name);
    if (S_ISDIR(fileinfo.st_mode)) {
      strcat(fname, STRSDIR);
    }
    if (!al_insert(sortlist, (char *) &fname))
      xmpi_fail((char*) "xmpi (al_insert)");
  }
/*
 * Loop adding the sorted files to the scrolled list.
 */
  p = (char **) al_top(sortlist);
  while (p) {
    xstr = XmStringCreateSimple(*p);
    XmListAddItem(w, xstr, 0);
    XmStringFree(xstr);
    free(*p);
    p = (char **) al_next(sortlist, (char *) p);
  }

  al_free(sortlist);
  closedir(pdir);
}

/*
 *	fnamecmp
 *
 *	Function:	- compare two filename entries in the sorted list
 *			- special cases: "./" is first, "../" is second
 *	Accepts:	- ptr to the two filenames
 *	Returns:	- -ve/0/+ve (the usual <, ==, > comparison)
 */
static int
fnamecmp(void *f1, void *f2)
{
  int ret;

  if (strcmp(*((char**) f1), "./") == 0) {
    ret = -1;
  } else if (strcmp(*((char**) f2), "./") == 0) {
    ret = 1;
  } else if (strcmp(*((char**) f1), "../") == 0) {
    ret = -1;
  } else if (strcmp(*((char**) f2), "../") == 0) {
    ret = 1;
  } else {
    ret = strcmp(*((char**) f1), *((char**) f2));
  }

  return (ret);
}
