/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: lamtr_util.c,v 1.3 1999/11/11 05:31:19 arodrig6 Exp $
 *
 *	Function:	- trace parsing utilities
 */

#include <errno.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>

#include "app_mgmt.h"
#include "lam.h"
#include "lamtrace.h"
#include "mpitrace.h"

/*
 * local variables
 */
static FILE		*trace_stream = 0;

/*
 *	lamtr_set_stream
 *
 *	Function:	- set trace input stream
 *	Accepts:	- open stream
 */
void
lamtr_set_stream(traces)

FILE			*traces;

{
	if (trace_stream) {
		fclose(trace_stream);
	}

	trace_stream = traces;
}

/*
 *	lamtr_open_file
 *
 *	Function:	- open a trace file
 *	Accepts:	- file name
 *	Returns:	- 0 or LAMERROR
 */
int
lamtr_open_file(tracefile)

char			*tracefile;

{
	FILE		*traces;

	if ((traces = fopen(tracefile, "r")) == 0) {
		return(LAMERROR);
	}

	lamtr_set_stream(traces);
	return(0);
}

/*
 *	lamtr_read
 *
 *	Function:	- read from the trace stream
 *	Accepts:	- buffer to put data read
 *			- size of an element
 *			- number of elements to read
 *	Returns:	- 0 for success, 1 on eof, else -1
 */
int
lamtr_read(ptr, size, count)

void			*ptr;
size_t			size;
size_t			count;

{
	size_t		nread;			/* number of elements read */

	if (trace_stream == 0) {
		return(-1);
	}

	nread = fread((char *) ptr, size, count, trace_stream);

	if (nread == 0 && ferror(trace_stream)) {
		return(LAMERROR);
	}
	else if (nread != count) {
		return(1);
	}
	else {
		return(0);
	}
}

/*
 *	lamtr_parse_world
 *
 *	Function:	- parse world trace
 *			- storage is allocated to hold the GPS array
 *			- values are returned in local byte order
 *	Accepts:	- size of world (out)
 *			- world GPS array (out)
 *	Returns:	- 0 or LAMERROR
 */
int
lamtr_parse_world(np, world)

int4			*np;
struct _gps		**world;

{
	int		magic;			/* magic number */
/*
 * Check the magic number.
 */
	if (lamtr_read(&magic, sizeof(int4), 1)) {
		return(LAMERROR);
	}

	ttoli4((int4 *) &magic, (int4 *) &magic);
	if (magic != LAM_TRMAGIC) {
		errno = EMAGIC;
		return(LAMERROR);
	}
/*
 * Get the size of the world.
 */
	if (lamtr_read(np, sizeof(int4), 1)) {
		return(LAMERROR);
	}
	ttoli4(np, np);
	if (*np <= 0) {
		errno = EINVAL;
		return(LAMERROR);
	}
/*
 * Allocate and read in GPS array.
 */
	*world = (struct _gps *) malloc((*np) * sizeof(struct _gps));
	if (*world == 0) return(LAMERROR);

	if (lamtr_read(*world, sizeof(struct _gps), (size_t) *np)) {
		return(LAMERROR);
	}

	mttoli4((int4 *) *world, (*np) * sizeof(struct _gps) / sizeof(int4));
	return(0);
}

/*
 *	lamtr_raw_world
 *
 *	Function:	- get raw world trace
 *			- storage is allocated to hold the GPS array
 *			- values are returned in LAM byte order
 *	Accepts:	- size of world (out)
 *			- world GPS array (out)
 *	Returns:	- 0 or LAMERROR
 */
int
lamtr_raw_world(np, world)

int4			*np;
char			**world;

{
	int4		n;			/* size of world */

	if (lamtr_read(np, sizeof(int4), 1)) {
		return(LAMERROR);
	}

	ttoli4(np, &n);

	*world = (char *) malloc(n * sizeof(struct _gps));
	if (*world == 0) return(LAMERROR);

	if (lamtr_read(*world, sizeof(struct _gps), (size_t) n)) {
		return(LAMERROR);
	}
	return(0);
}

/*
 *	lamtr_parse_comm
 *
 *	Function:	- parse a communicator trace
 *			- storage is allocated to hold the GPS array
 *	Accepts:	- context ID (out)
 *			- size local group (out)
 *			- size remote group (out)
 *			- groups GPS array (out)
 *	Returns:	- 0 or LAMERROR
 */
int
lamtr_parse_comm(cid, nlocal, nremote, grps)

int4			*cid;
int4			*nlocal;
int4			*nremote;
struct _gps		**grps;

{
	struct trcid	cid_hdr;		/* cid trace head */
	size_t 		nbytes;			/* size of GPS array */
/*
 * Read the trace head.
 */
	if (lamtr_read(&cid_hdr, sizeof(struct trcid), 1)) {
		return(LAMERROR);
	}
	mttoli4((int4 *) &cid_hdr, sizeof(struct trcid) / sizeof(int4));
/*
 * Extract communicator id and group sizes.
 */
	*cid = cid_hdr.trc_cid;
	*nlocal = cid_hdr.trc_nlg;
	*nremote = cid_hdr.trc_nrg;

	nbytes = (*nlocal + *nremote) * sizeof(struct _gps);
/*
 * Allocate the group GPS array and read it in.
 */
	if ((*grps = (struct _gps *) malloc(nbytes)) == 0) {
		return(LAMERROR);
	}

	if (lamtr_read(*grps, nbytes, 1)) {
		free(*grps);
		return(LAMERROR);
	}

	mttoli4(*grps, nbytes / sizeof(int4));
	return(0);
}

/*
 *	lamtr_raw_comm
 *
 *	Function:	- get raw communicator trace
 *			- storage is allocated for the groups GPS array
 *			- values are unconverted
 *	Accepts:	- communicator trace header (out)
 *			- groups GPS array (out)
 *	Returns:	- 0 or LAMERROR
 */
int
lamtr_raw_comm(cid, grps)

char			*cid;
struct _gps		**grps;

{
	size_t		nbytes;			/* size of GPS array */
	int4		nlocal;			/* local group size */
	int4		nremote;		/* remote group size */
/*
 * Read the communicator trace header.
 */
	if (lamtr_raw_cid_m(*cid)) {
		return(LAMERROR);
	}
/*
 * Extract the group sizes.
 */
	ttoli4(&((struct trcid *) cid)->trc_nlg,  &nlocal);
	ttoli4(&((struct trcid *) cid)->trc_nrg,  &nremote);

	nbytes = (nlocal + nremote) * sizeof(struct _gps);
/*
 * Allocate the group GPS array and read it in.
 */
	if ((*grps = (struct _gps *) malloc((size_t) nbytes)) == 0) {
		return(LAMERROR);
	}

	if (lamtr_read(*grps, nbytes, 1)) {
		free(*grps);
		return(LAMERROR);
	}

	return(0);
}

/*
 *	lamtr_raw_dtype
 *
 *	Function:	- get raw datatype trace
 *			- values are unconverted
 *			- storage is allocated for the trace
 *	Accepts:	- raw datatype trace (out)
 *	Returns:	- 0 or LAMERROR
 */
int
lamtr_raw_dtype(dtype)

char			**dtype;

{
	struct trdtype	dttr;			/* datatype trace head */
	int4		len;			/* trace length */
/*
 * Read the trace head.
 */
	if (lamtr_read(&dttr, sizeof(struct trdtype), 1)) {
		return(LAMERROR);
	}
	ttoli4(&dttr.trd_length, &len);
/*
 * Allocate the datatype buffer (flattened type map) and read it in.
 */
	if ((*dtype = (char *) malloc((size_t) len)) == 0) {
		return(LAMERROR);
	}

	if (lamtr_read((*dtype) + sizeof(struct trdtype),
				len - sizeof(struct trdtype), 1)) {
		free(*dtype);
		return(LAMERROR);
	}

	memcpy(*dtype, (char *) &dttr, sizeof(struct trdtype));
	return(0);
}

/*
 *	lamtr_parse_dtype
 *
 *	Function:	- parse a datatype trace
 *			- values are returned in local byte order
 *			- storage is allocated for the trace
 *	Accepts:	- datatype label (out)
 *			- datatype trace (out)
 *	Returns:	- 0 or LAMERROR
 */
int
lamtr_parse_dtype(dtype, dtbuf)

int4			*dtype;
char			**dtbuf;

{
	int		len;			/* trace length */

	if (lamtr_raw_dtype(dtbuf)) {
		return(LAMERROR);
	}

	ttoli4(&((struct trdtype *) *dtbuf)->trd_length, &len);
	mttoli4((int4 *) *dtbuf, len / sizeof(int4));
	len=len;
	*dtype = ((struct trdtype *) *dtbuf)->trd_dtype;
	return(0);
}

/*
 *	lamtr_find_rank
 *
 *	Function:	- finds global rank corresponding to node and pid
 *	Accepts:	- size of world
 *			- world GPS array
 *			- node
 *			- pid
 *	Returns:	- global rank corresponding to node and pid, else (-1)
 */
int
lamtr_find_rank(np, world, node, pid)

int			np;
struct _gps		*world;
int			node;
int			pid;

{
	int		i;

	for (i = 0; i < np; i++, world++) {
		if (world->gps_node == node && world->gps_pid == pid) {
			return(i);
		}
	}

	return(-1);
}
