// file      : xsde/cxx/elements.hxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#ifndef CXX_ELEMENTS_HXX
#define CXX_ELEMENTS_HXX

#include <cult/types.hxx>
#include <cult/containers/map.hxx>
#include <cult/containers/vector.hxx>

#include <xsd-frontend/semantic-graph.hxx>
#include <xsd-frontend/traversal.hxx>

#include <elements.hxx>

#include <locale>
#include <ostream>

namespace CXX
{
  using std::endl;
  typedef WideString String;


  // On some platforms std::toupper can be something other than a
  // function with C++ linkage.
  //
  wchar_t
  upcase (wchar_t c);


  // Exceptions.
  //

  struct NoNamespaceMapping
  {
    NoNamespaceMapping (SemanticGraph::Path const& file,
                        UnsignedLong line,
                        UnsignedLong column,
                        String const& ns)
        : file_ (file),
          line_ (line),
          column_ (column),
          ns_ (ns)
    {
    }


    SemanticGraph::Path const&
    file () const
    {
      return file_;
    }

    UnsignedLong
    line () const
    {
      return line_;
    }

    UnsignedLong
    column () const
    {
      return column_;
    }

    String const&
    ns () const
    {
      return ns_;
    }

  private:
    SemanticGraph::Path file_;
    UnsignedLong line_;
    UnsignedLong column_;
    String ns_;
  };

  struct InvalidNamespaceMapping
  {
    InvalidNamespaceMapping (String const& mapping,
                             String const& reason)
        : mapping_ (mapping), reason_ (reason)
    {
    }

    String const&
    mapping () const
    {
      return mapping_;
    }

    String const&
    reason () const
    {
      return reason_;
    }

  private:
    String mapping_;
    String reason_;
  };


  //
  //
  class Context
  {
  public:
    typedef Cult::Containers::Vector<String> RegexMapping;
    typedef Cult::Containers::Map<String, String> MapMapping;
    typedef Cult::Containers::Map<String, String> MappingCache;

  public:
    Context (std::wostream& o,
             SemanticGraph::Schema& root,
             NarrowString const& char_type__,
             Boolean include_with_brackets__,
             NarrowString const& include_prefix__,
             NarrowString const& esymbol__,
             Containers::Vector<NarrowString> const& nsm,
             Containers::Vector<NarrowString> const& nsr,
             Boolean trace_namespace_regex_,
             Containers::Vector<NarrowString> const& include_regex,
             Boolean trace_include_regex_,
             Boolean inline_);

  protected:
    Context (Context& c)
        : os (c.os),
          schema_root (c.schema_root),
          char_type (c.char_type),
          L (c.L),
          string_type (c.string_type),
          include_with_brackets (c.include_with_brackets),
          include_prefix (c.include_prefix),
          esymbol (c.esymbol),
          inl (c.inl),
          ns_mapping_cache (c.ns_mapping_cache),
          trace_namespace_regex (c.trace_namespace_regex),
          nsr_mapping (c.nsr_mapping),
          nsm_mapping (c.nsm_mapping),
          include_mapping (c.include_mapping),
          trace_include_regex (c.trace_include_regex)
    {
    }

    Context (Context& c, std::wostream& o)
        : os (o),
          schema_root (c.schema_root),
          char_type (c.char_type),
          L (c.L),
          string_type (c.string_type),
          include_with_brackets (c.include_with_brackets),
          include_prefix (c.include_prefix),
          esymbol (c.esymbol),
          inl (c.inl),
          ns_mapping_cache (c.ns_mapping_cache),
          trace_namespace_regex (c.trace_namespace_regex),
          nsr_mapping (c.nsr_mapping),
          nsm_mapping (c.nsm_mapping),
          include_mapping (c.include_mapping),
          trace_include_regex (c.trace_include_regex)
    {
    }

  public:
    static String
    unclash (String const& name, String const& new_name)
    {
      return name == new_name ? (new_name + L'_') : new_name;
    }

  public:

    // Escape C++ keywords and illegal characters.
    //
    static String
    escape (String const&);

    // Escape a string so that it can be used as body of a C++ string
    // literal (i.e., put between "").
    //
    static String
    escape_str (String const&);

    // Translate XML namespace name to a C++ identifier.
    //
    String
    ns_name (SemanticGraph::Namespace&);

    // C++ namespace for XML Schema.
    //
    String
    xs_ns_name ();

    //
    //
    SemanticGraph::Namespace&
    namespace_ (SemanticGraph::Nameable& n);

    // Original XML namespace name.
    //
    String
    xml_ns_name (SemanticGraph::Nameable& ns);


    // Fully-qualified C++ name.
    //
    String
    fq_name (SemanticGraph::Nameable& n, Char const* name_key = "name");

  public:
    String
    process_include_path (String const&) const;
    
  public:
    // Get escaped name.
    //
    static String const&
    ename (SemanticGraph::Nameable const& n)
    {
      return n.context ().get<String> ("name");
    }

  public:
    std::wostream& os;

    SemanticGraph::Schema& schema_root;

    String& char_type;
    String& L;                  // string literal prefix
    String& string_type;

    Boolean& include_with_brackets;
    String& include_prefix;

    String& esymbol;
    String& inl;

    static std::locale locale;

  public:
    MappingCache& ns_mapping_cache;

  private:
    String char_type_;
    String L_;
    String string_type_;

    Boolean include_with_brackets_;
    String include_prefix_;

    String esymbol_;
    String inl_;

  private:
    Boolean trace_namespace_regex;
    RegexMapping nsr_mapping_;
    MapMapping nsm_mapping_;
    RegexMapping const& nsr_mapping;
    MapMapping const& nsm_mapping;
    MappingCache ns_mapping_cache_;

    RegexMapping include_mapping_;
    RegexMapping const& include_mapping;
    Boolean trace_include_regex;
  };


  // Usual namespace mapping.
  //
  struct Namespace : Traversal::Namespace
  {
    struct ScopeTracker
    {
      virtual Void
      enter (String const&) = 0;

      virtual Void
      leave () = 0;
    };


    Namespace (Context& c, Boolean set_scope = false)
        : ctx_ (c), st_ (0)
    {
    }

    Namespace (Context& c, ScopeTracker& st)
        : ctx_ (c), st_ (&st)
    {
    }

    virtual Void
    pre (Type&);

    virtual Void
    post (Type&);

  private:
    Context& ctx_;
    ScopeTracker* st_;
  };


  //
  //
  template <typename X>
  struct Has : X
  {
    Has (Boolean& result)
        : result_ (result)
    {
    }

    virtual Void
    traverse (typename X::Type&)
    {
      result_ = true;
    }

  private:
    Boolean& result_;
  };

  // Checks if scope 'Y' names any of 'X'
  //
  template <typename X, typename Y>
  Boolean
  has (Y& y)
  {
    using SemanticGraph::Scope;

    Boolean result (false);
    Has<X> t (result);

    for (Scope::NamesIterator i (y.names_begin ()), e (y.names_end ());
         !result && i != e; ++i)
      t.dispatch (i->named ());

    return result;
  }

  // Checks if the compositor has any particle of 'X'
  //
  template <typename X>
  Boolean
  has_particle (SemanticGraph::Compositor& y)
  {
    using SemanticGraph::Compositor;

    Boolean result (false);
    Has<X> t (result);

    for (Compositor::ContainsIterator i (y.contains_begin ()),
           e (y.contains_end ()); !result && i != e; ++i)
    {
      SemanticGraph::Particle& p (i->particle ());

      t.dispatch (p);

      if (!result && p.is_a<Compositor> ())
        result = has_particle<X> (dynamic_cast<Compositor&> (p));
    }

    return result;
  }

  // Specialization for Complex
  //
  template <typename X>
  Boolean
  has_particle (SemanticGraph::Complex& c)
  {
    return c.contains_compositor_p () &&
      has_particle<X> (c.contains_compositor ().compositor ());
  }

  // Fundamental type mapping helper.
  //
  struct Fundamental : Traversal::Fundamental::Type,
                       Traversal::Fundamental::String,
                       Traversal::Fundamental::NormalizedString,
                       Traversal::Fundamental::Token,
                       Traversal::Fundamental::Name,
                       Traversal::Fundamental::NameToken,
                       Traversal::Fundamental::NCName,
                       Traversal::Fundamental::Id,
                       Traversal::Fundamental::IdRef
  {
    virtual Void
    fundamental_type (SemanticGraph::Fundamental::Type& t) = 0;

    virtual Void
    fundamental_template (SemanticGraph::Fundamental::Type& t) = 0;

    virtual Void
    traverse (SemanticGraph::Fundamental::Type& t)
    {
      fundamental_type (t);
    }

    virtual Void
    traverse (SemanticGraph::Fundamental::String& t)
    {
      fundamental_template (t);
    }

    virtual Void
    traverse (SemanticGraph::Fundamental::NormalizedString& t)
    {
      fundamental_template (t);
    }

    virtual Void
    traverse (SemanticGraph::Fundamental::Token& t)
    {
      fundamental_template (t);
    }

    virtual Void
    traverse (SemanticGraph::Fundamental::Name& t)
    {
      fundamental_template (t);
    }

    virtual Void
    traverse (SemanticGraph::Fundamental::NameToken& t)
    {
      fundamental_template (t);
    }

    virtual Void
    traverse (SemanticGraph::Fundamental::NCName& t)
    {
      fundamental_template (t);
    }

    virtual Void
    traverse (SemanticGraph::Fundamental::Id& t)
    {
      fundamental_template (t);
    }

    virtual Void
    traverse (SemanticGraph::Fundamental::IdRef& t)
    {
      fundamental_template (t);
    }
  };

  //
  //
  struct Includes : Traversal::Imports,
                    Traversal::Includes
  {
    Includes (Context& c, String const& expr)
        : ctx_ (c), expr_ (expr)
    {
    }

    virtual Void
    traverse (SemanticGraph::Imports& i)
    {
      traverse (i.path ());
    }

    virtual Void
    traverse (SemanticGraph::Includes& i)
    {
      traverse (i.path ());
    }

    Void
    traverse (SemanticGraph::Path const&);

  private:
    Context& ctx_;
    String expr_;
  };
}

#endif  // CXX_TREE_ELEMENTS_HXX
