/*******************************************************************
 * The driver function for a Linux application layer EAPOL 
 * implementation
 * File: eapmd5.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapmd5.c,v 1.21 2005/10/17 03:56:54 chessing Exp $
 * $Date: 2005/10/17 03:56:54 $
 * $Log: eapmd5.c,v $
 * Revision 1.21  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.20  2005/08/09 01:39:16  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <openssl/ssl.h>
#include <openssl/md5.h>
#include <string.h>
#include <strings.h>

#include "xsup_debug.h"
#include "xsup_err.h"
#include "frame_structs.h"
#include "profile.h"
#include "xsupconfig.h"   
#include "eap.h"
#include "eapmd5.h"

#define MD5_LENGTH    0x10

/*****************************************************
 *
 * Setup to handle MD5 EAP requests
 *
 * This function is called each time we recieve a packet of the EAP type MD5.
 * At a minimum, it should check to make sure it's stub in the structure 
 * exists, and if not, set up any variables it may need.  Since MD5 doesn't
 * have any state that needs to survive successive calls, we don't need to 
 * do anything here.
 *
 *****************************************************/
int eapmd5_setup(struct generic_eap_data *thisint)
{
  // Do anything special that might be needed for this EAP type to work.
  debug_printf(DEBUG_EVERYTHING, "Initalized EAP-MD5!\n");

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid EAP structure passed in to eapmd5_setup()!\n");
      return XEMALLOC;
    }

  thisint->eap_data = (int *)malloc(sizeof(int));
  if (thisint->eap_data == NULL) return XEMALLOC;

  return XENONE;
}


/*****************************************************
 *
 * Process MD5 EAP Requests
 *
 *
 ******************************************************/
int eapmd5_process(struct generic_eap_data *thisint, u_char *dataoffs, 
		   int insize, u_char *outframe, int *outsize)
{
  struct md5_values *md5data, *md5out;
  struct config_eap_md5 *userdata;
  u_char md5_result[16];
  u_char *tohash;
  char *username;
  int tohashlen;
  int *processReady;

  if (!thisint->eap_data)
    {
      debug_printf(DEBUG_NORMAL, "Invalid EAP data passed in to eapmd5_process()!\n");
      return XEMALLOC;
    }

  if (!outframe)
    {
      debug_printf(DEBUG_NORMAL, "Invalid buffer for return data in eapmd5_process()!\n");
      return XEMALLOC;
    }

  processReady = (int *)thisint->eap_data;
  userdata = (struct config_eap_md5 *)thisint->eap_conf_data;

  debug_printf(DEBUG_EVERYTHING, "(EAP-MD5) Processing.\n");

  if ((thisint->tempPwd == NULL) && (userdata->password == NULL))
    {
      thisint->need_password = 1;
      thisint->eaptype = strdup("EAP-MD5");
      thisint->eapchallenge = NULL;

      *outsize = 0;
      return XENONE;
    }

  // Make sure we have something to process...
  if (dataoffs == NULL) return XENONE;

  if (userdata->username == NULL)
    {
      username = thisint->identity;
    } else {
      username = userdata->username;
    }

  if ((userdata->password == NULL) && (thisint->tempPwd != NULL))
    {
      userdata->password = thisint->tempPwd;
      thisint->tempPwd = NULL;
      /*      debug_printf(DEBUG_AUTHTYPES, "Passed in password : %s\n",
	      userdata->password);*/
    }

  // Actually process, and respond to challenges.
  md5data = (struct md5_values *)dataoffs;
  md5out = (struct md5_values *)outframe;
  
  if (md5data->length != MD5_LENGTH)
    {
      debug_printf(DEBUG_NORMAL, "(EAP-MD5) Incorrect length value for MD5 random value.\n");
      return XEMD5LEN;
    }

  tohashlen = (1+md5data->length+strlen(userdata->password));
  tohash = (u_char *)malloc(tohashlen);
  if (tohash == NULL)
    {
      debug_printf(DEBUG_NORMAL, "(EAP-MD5) Couldn't allocate memory for building hash source!\n");
      return XEMALLOC;
    }

  // Make sure we clean out the memory space.
  bzero(tohash, tohashlen);
  
  // Build the information we need to hash. Start with the EAP identifier.
  tohash[0] = thisint->eapid;

  // Then, we need the password.
  memcpy(&tohash[1], userdata->password, strlen(userdata->password));
  
  // Then the random value sent to us.
  memcpy(&tohash[1+strlen(userdata->password)], &md5data->randval, MD5_LENGTH);

  // Now, run it through the hash routine.
  MD5(tohash, tohashlen, &md5_result[0]);

  // We are done with tohash, so free it.
  free(tohash);

  // Set up our response frame.
  md5out->length = MD5_LENGTH;
  memcpy(&md5out->randval[0], &md5_result[0], MD5_LENGTH);

  memcpy(&outframe[sizeof(struct md5_values)], username, strlen(username));

  *outsize = (sizeof(struct md5_values)+strlen(username));

  return XENONE;
}

/*******************************************************
 *
 * Set our keys, if we can.
 *
 *******************************************************/
int eapmd5_get_keys(struct interface_data *thisint)
{
  // We don't key, so return -1.  (We return 0 if we set a key.)
  return -1;
}

/*******************************************************
 *
 * Clean up after ourselves.  This will get called when we get a packet that
 * needs to be processed requests a different EAP type.  It will also be 
 * called on termination of the program.
 *
 *******************************************************/
int eapmd5_cleanup(struct generic_eap_data *thisint)
{
  // Clean up after ourselves.

  debug_printf(DEBUG_AUTHTYPES, "(EAP-MD5) Cleaning up.\n");

  if (thisint->eap_data != NULL)
    {
      free(thisint->eap_data);
      thisint->eap_data = NULL;
    }

  return XENONE;
}

/*******************************************************
 *
 * If we fail an authentication, we will call this routine.  It should clean
 * up anything that shouldn't live in to the next authentication attempt.
 *
 *******************************************************/
int eapmd5_failed(struct generic_eap_data *thisint)
{
  struct config_eap_md5 *userdata;

  if ((thisint == NULL) || (thisint->eap_conf_data == NULL))
    {
      debug_printf(DEBUG_AUTHTYPES, "Invalid MD5 configuration data!\n");
      return XEMALLOC;
    }

  userdata = (struct config_eap_md5 *)thisint->eap_conf_data;

  // If configure was passed the no password reset flag, we shouldn't do
  // anything!
#ifndef NO_PWD_RESET
  /*
  if (userdata->password != NULL)
    {
      free(userdata->password);
      userdata->password = NULL;
    }
  */
#endif

  return XENONE;
}
