/*******************************************************************
 * Handle state machine related functions.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 * File: statemachine.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: statemachine.c,v 1.58 2006/02/23 22:26:53 chessing Exp $
 * $Date: 2006/02/23 22:26:53 $
 * $Log: statemachine.c,v $
 * Revision 1.58  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.57  2006/01/23 20:33:47  chessing
 * Added support for the replay counters in WPA/WPA2.  BSSID is now selected based on the signal quality information returned during the scan.  (Assuming signal quality information is returned during a scan. ;)
 *
 * Revision 1.56  2006/01/03 04:02:35  chessing
 * Added the ability to store the PEAP password in a hashed format.  (Basically, an MS-CHAPv1 hash.)  Also added an 'ntpwdhash' program to the tools directory that will convert a cleartext password in to a hash that can be copied to the configuration file.
 *
 * Revision 1.55  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.54  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.53  2005/10/13 19:03:15  chessing
 * Small statemachine fix.
 *
 * Revision 1.52  2005/10/13 18:46:47  chessing
 * Fixed to the Madwifi driver to allow it to do dynamic WEP again.  Fixed up the wext driver to behave correctly again. ;)
 *
 * Revision 1.51  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.50  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.49  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.48  2005/08/20 19:06:53  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.47  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netinet/in.h>
#include <strings.h>

#include "profile.h"
#include "xsupconfig.h"
#include "snmp.h"
#include "statemachine.h"
#include "xsup_debug.h"
#include "frame_structs.h"
#include "eap.h"
#include "eapol.h"
#include "xsup_err.h"
#include "xsup_ipc.h"
#include "wpa.h"
#include "backend_sm.h"
#include "cardif/cardif.h"
#include "eap_types/leap/eapleap.h"
#include "timer.h"
#include "wireless_sm.h"

// This is needed to keep the compiler from complaining about not knowing where
// global_deinit() is.  We should probably change this to something a little
// more "correct".
extern void global_deinit();

/******************************************
 *
 * Decrement a value, as long as it is greater than 0.
 *
 ******************************************/
void dec_if_nz(char *decval)
{
  if (!decval) return;

  if (*decval > 0) (*decval)--;
}

/******************************************
 *
 * Initalize the state machine
 *
 ******************************************/
int statemachine_init(struct interface_data *newint)
{
  if (!newint) 
    {
      debug_printf(DEBUG_NORMAL, "newint == NULL in %s:%d!\n", __FUNCTION__,
		   __LINE__);
      return XEMALLOC;
    }

  newint->statemachine = (struct dot1x_state *)malloc(sizeof(struct dot1x_state));
  if (newint->statemachine == NULL) return XEMALLOC;

  statemachine_reinit(newint);

  backend_sm_init(newint);

  return XENONE;
}

/******************************************
 *
 * Reinitalize the state machine
 *
 ******************************************/
int statemachine_reinit(struct interface_data *newint)
{
  if (!newint) 
    {
      debug_printf(DEBUG_NORMAL, "newint == NULL in %s:%d!\n", __FUNCTION__,
		   __LINE__);
      return XEMALLOC;
    }

  debug_printf(DEBUG_STATE, "Reinit state machine\n");

  newint->statemachine->initialize = 0;

  // Now, we want to set up a few defaults as per the 802.1x doc, and
  // initalize a few other statemachine variables that we will be needing.
  newint->statemachine->authPeriod = 30;
  newint->statemachine->authWhile = newint->statemachine->authPeriod;

  newint->statemachine->heldPeriod = 60;
  newint->statemachine->heldWhile = newint->statemachine->heldPeriod;

  newint->statemachine->startPeriod = 30;
  newint->statemachine->startWhen = 0;     // Trigger sending an EAPOL-Start
  newint->statemachine->maxStart = 3;

  // Set up our inital state.
  newint->statemachine->userLogoff = FALSE;
  newint->statemachine->logoffSent = FALSE;
  newint->statemachine->eapFail = FALSE;
  newint->statemachine->startCount = 0;

  newint->statemachine->tick = FALSE;

  /* Some new variables defined by 802.1X-2004 */
  newint->statemachine->eapFail = FALSE;
  newint->statemachine->eapolEap = FALSE;
  newint->statemachine->eapSuccess = FALSE;
  newint->statemachine->keyAvailable = FALSE;
  newint->statemachine->keyDone = FALSE;
  newint->statemachine->keyRun = FALSE;
  newint->statemachine->keyTxEnabled = FALSE;
  newint->statemachine->portControl = AUTO;
  newint->statemachine->portEnabled = FALSE;
  newint->statemachine->portValid = TRUE;
  newint->statemachine->suppAbort = FALSE;
  newint->statemachine->suppFail = FALSE;
  newint->statemachine->suppPortStatus = UNAUTHORIZED;
  newint->statemachine->suppStart = FALSE;
  newint->statemachine->suppSuccess = FALSE;
  newint->statemachine->suppTimeout = FALSE;

  newint->statemachine->eapRestart = FALSE;
  newint->statemachine->logoffSent = FALSE;
  newint->statemachine->sPortMode = AUTO;
  newint->statemachine->startCount = 0;

  /* Init 802.11i-D3.0 vars. */
  newint->statemachine->DeauthenticationRequest = FALSE;
  newint->statemachine->AuthenticationRequest = FALSE;
  newint->statemachine->AuthenticationFailed = FALSE;
  newint->statemachine->EAPOLKeyReceived = FALSE;
  newint->statemachine->IntegrityFailed = FALSE;
  newint->statemachine->MICVerified = FALSE;
  newint->statemachine->Counter = 0;
  newint->statemachine->SNonce = NULL;
  newint->statemachine->PTK = NULL;
  newint->statemachine->TPTK = NULL;
  newint->statemachine->GTK = NULL;
  newint->statemachine->PMK = NULL;

  newint->statemachine->MICfailures = 0;

  bzero(newint->statemachine->replay_counter, 6);

  // The 0 below is the DISCONNECTED state.
  newint->statemachine->wpaCurState = 0;
  newint->statemachine->wpaLastState = -1;

  newint->statemachine->curState = DISCONNECTED;

  newint->statemachine->unicastKeyLen = 0;
  newint->statemachine->broadcastKeyLen = 0;

  backend_sm_reinit(newint);

  return XENONE;
}

/****************************************************************
 *
 * Display the state requested.
 *
 ****************************************************************/
void statemachine_disp_state(int debuglevel, int state)
{
  switch (state)
    {
    case DISCONNECTED:
      debug_printf_nl(debuglevel, "DISCONNECTED");
      break;
    case LOGOFF:
      debug_printf_nl(debuglevel, "LOGOFF");
      break;
    case ACQUIRED:   // No longer part of 802.1X in 802.1X-2004.
      debug_printf_nl(debuglevel, "ACQUIRED");
      break;
    case AUTHENTICATING:
      debug_printf_nl(debuglevel, "AUTHENTICATING");
      break;
    case AUTHENTICATED:
      debug_printf_nl(debuglevel, "AUTHENTICATED");
      break;
    case CONNECTING:
      debug_printf_nl(debuglevel, "CONNECTING");
      break;
    case HELD:
      debug_printf_nl(debuglevel, "HELD");
      break;
    case RESTART:
      debug_printf_nl(debuglevel, "RESTART");
      break;
    case S_FORCE_AUTH:
      debug_printf_nl(debuglevel, "S_FORCE_AUTH");
      break;
    case S_FORCE_UNAUTH:
      debug_printf_nl(debuglevel, "S_FORCE_UNAUTH");
      break;
    default:
      debug_printf_nl(debuglevel, "UNKNOWN!");
      break;
    }
}

/****************************************************************
 *
 * Change state to force authorized state.
 *
 ****************************************************************/
int statemachine_change_to_s_force_auth(struct interface_data *ctx)
{
  // Display the state we are changing to.
  debug_printf(DEBUG_STATE, "(global) -> S_FORCE_AUTH\n");

  // Verify that we have valid parameters.
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  // Make the change to the new state, and notify any listening GUIs.
  ctx->statemachine->curState = S_FORCE_AUTH;
  xsup_ipc_send_auth_state(ctx);

  ctx->statemachine->suppPortStatus = AUTHORIZED;
  ctx->statemachine->sPortMode = FORCE_AUTHORIZED;

  return XENONE;
}

/******************************************************************
 *
 * Do this while in force_authorized state.
 *
 ******************************************************************/
void statemachine_do_s_force_auth(struct interface_data *ctx)
{
  // Just take a nap to keep from wasting time on the CPU.
  usleep(500000);
}

/****************************************************************
 *
 * Change state to force authorized state.
 *
 ****************************************************************/
int statemachine_change_to_s_force_unauth(struct interface_data *ctx)
{
  // Display the state we are changing to.
  debug_printf(DEBUG_STATE, "(global) -> S_FORCE_UNAUTH\n");

  // Verify that we have valid parameters.
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  // Make the change to the new state, and notify any listening GUIs.
  ctx->statemachine->curState = S_FORCE_UNAUTH;
  xsup_ipc_send_auth_state(ctx);

  ctx->statemachine->suppPortStatus = UNAUTHORIZED;
  ctx->statemachine->sPortMode = FORCE_UNAUTHORIZED;

  // Send a logoff per 802.1X-2004 8.2.11.10
  txLogoff(ctx);

  return XENONE;
}

/******************************************************************
 *
 * Do this while in force_unauthorized state.
 *
 ******************************************************************/
void statemachine_do_s_force_unauth(struct interface_data *ctx)
{
  // Just take a nap to keep from wasting time on the CPU.
  usleep(500000);
}

/******************************************************************
 *
 * Change to LOGOFF state.
 *
 ******************************************************************/
int statemachine_change_to_logoff(struct interface_data *ctx)
{
  // Verify that we came from a valid state.

  // Verify our parameters are correct.
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }
  txLogoff(ctx);
  ctx->statemachine->logoffSent = TRUE;
  ctx->statemachine->suppPortStatus = UNAUTHORIZED;
  ctx->statemachine->curState = LOGOFF;
  xsup_ipc_send_auth_state(ctx);

  return XENONE;
}

/******************************************************************
 *
 * Process LOGOFF state.
 *
 ******************************************************************/
void statemachine_do_logoff(struct interface_data *ctx)
{
  if (!ctx->statemachine->userLogoff)
    {
      statemachine_change_state(ctx, DISCONNECTED);
    } else {
      usleep(500000);
    }
}

/******************************************************************
 *
 * Change to disconnected state.
 *
 ******************************************************************/
int statemachine_change_to_disconnected(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid statemachine context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  ctx->statemachine->sPortMode = AUTO;
  ctx->statemachine->startCount = 0;
  ctx->statemachine->logoffSent = FALSE;
  ctx->statemachine->suppPortStatus = UNAUTHORIZED;
  ctx->statemachine->suppAbort = TRUE;

  ctx->statemachine->curState = DISCONNECTED;
  xsup_ipc_send_auth_state(ctx);

  return XENONE;
}

/******************************************************************
 *
 * Process disconnected state.
 *
 ******************************************************************/
void statemachine_do_disconnected(struct interface_data *ctx)
{
  if ((!ctx->statemachine->initialize) &&
      (ctx->statemachine->portEnabled))
    {
      statemachine_change_state(ctx, CONNECTING);
    } else {
      usleep(500000);
    }
}

/******************************************************************
 *
 * Change to held state.
 *
 ******************************************************************/
int statemachine_change_to_held(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid statemachine context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if ((ctx->statemachine->curState != CONNECTING) &&
      (ctx->statemachine->curState != AUTHENTICATING))
    {
      debug_printf(DEBUG_NORMAL, "Invalid attempt to change to HELD state! "
		   "Going to DISCONNECTED state to attempt to reset the "
		   "state machine!\n");
      return statemachine_change_to_disconnected(ctx);
    }
      
  ctx->statemachine->heldWhile = ctx->statemachine->heldPeriod;
  ctx->statemachine->suppPortStatus = UNAUTHORIZED;
  ctx->statemachine->curState = HELD;
  xsup_ipc_send_auth_state(ctx);

  return XENONE;
}

/******************************************************************
 *
 * Process held state.
 *
 ******************************************************************/
void statemachine_do_held(struct interface_data *ctx)
{
  if (ctx->statemachine->heldWhile == 0)
    {
      statemachine_change_state(ctx, CONNECTING);
      return;
    }
  
  if (ctx->statemachine->eapolEap)
    {
      statemachine_change_state(ctx, RESTART);
      return;
    }

  usleep(500000);
}

/******************************************************************
 *
 * Change to connecting state.
 *
 ******************************************************************/
int statemachine_change_to_connecting(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  // We should only transition to this state from CONNECTING, HELD, 
  // DISCONNECTED, or AUTHENTICATING.
  if ((ctx->statemachine->curState != CONNECTING) &&
      (ctx->statemachine->curState != HELD) &&
      (ctx->statemachine->curState != DISCONNECTED) &&
      (ctx->statemachine->curState != AUTHENTICATING))
    {
      debug_printf(DEBUG_NORMAL, "Attempt to change to CONNECTING state "
		   "from an invalid state!!  Going to DISCONNECTED state "
		   "to attempt to reset the state machine.\n");
      return statemachine_change_state(ctx, DISCONNECTED);
    }

  // XXX This deviates from a strict interpretation of the IEEE 802.1X-2004
  // standard.  But, it seems silly to send a start message if the 
  // authenticator has already started a conversation.
  ctx->statemachine->curState = CONNECTING;

  if (ctx->statemachine->eapolEap)
    {
      // If we have an EAP message in the buffer, then skip sending a start,
      // and go directly to authenticating.
      statemachine_change_state(ctx, RESTART);
      return XENONE;
    }

  ctx->statemachine->startWhen = ctx->statemachine->startPeriod;
  ctx->statemachine->startCount = ctx->statemachine->startCount + 1;
  ctx->statemachine->eapolEap = FALSE;
  txStart(ctx);

  xsup_ipc_send_auth_state(ctx);

  return XENONE;
}

/******************************************************************
 *
 * Process connecting state.
 *
 ******************************************************************/
void statemachine_do_connecting(struct interface_data *ctx)
{
  if ((ctx->statemachine->startWhen == 0) &&
      (ctx->statemachine->startCount < ctx->statemachine->maxStart))
    {
      // Change to connecting state again.
      statemachine_change_state(ctx, CONNECTING);
      return;
    }
  
  if (((ctx->statemachine->startWhen == 0) &&
       (ctx->statemachine->startCount >= ctx->statemachine->maxStart)
       && ctx->statemachine->portValid))
    {
      // Here, we have defaulted in to authenticated state.  We should
      // display a message to the user to avoid confusion, and terminate
      // if we were asked to.
      debug_printf(DEBUG_NORMAL, "Xsupplicant has defaulted to "
		   "authenticated state, due to the inability to\n"
		   "successfully start/complete an EAP conversation.  It "
		   "is likely that this\nauthenticator doesn't support "
		   "802.1X, or that 802.1X isn't configured correctly\n"
		   "on the authenticator or RADIUS server.\n");

      statemachine_change_state(ctx, AUTHENTICATED);
      
      if (ctx->flags & TERM_ON_FAIL)
	{
	  // We have been instructed to terminate.  So, do it cleanly.
	  global_deinit();
	}
      return;
    }
  
  if (ctx->statemachine->eapSuccess || ctx->statemachine->eapFail)
    {
      statemachine_change_state(ctx, AUTHENTICATING);
      return;
    }
  
  if (ctx->statemachine->eapolEap)
    {
      statemachine_change_state(ctx, RESTART);
      return;
    }
  
  if ((ctx->statemachine->startWhen == 0) &&
      (ctx->statemachine->startCount >= ctx->statemachine->maxStart)
      && !ctx->statemachine->portValid)
    {
      statemachine_change_state(ctx, HELD);
      return;
    }

  usleep(500000);
}

/******************************************************************
 *
 * Change to restart state.
 *
 ******************************************************************/
int statemachine_change_to_restart(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }
  
  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  // We should only get here from AUTHENTICATED, CONNECTING, or HELD states.
  if ((ctx->statemachine->curState != AUTHENTICATED) &&
      (ctx->statemachine->curState != CONNECTING) &&
      (ctx->statemachine->curState != HELD))
    {
      debug_printf(DEBUG_NORMAL, "Attempt to change to RESTART state from "
		   "an invalid state!  Changing to DISCONNECTED state to "
		   "attempt to recover the state machine.\n");
      return statemachine_change_state(ctx, DISCONNECTED);
    }

  ctx->statemachine->eapRestart = TRUE;
  ctx->statemachine->curState = RESTART;
  xsup_ipc_send_auth_state(ctx);

  return XENONE;
}

/******************************************************************
 *
 * Process restart state.
 *
 ******************************************************************/
void statemachine_do_restart(struct interface_data *ctx)
{
  if (!ctx->statemachine->eapRestart)
    {
      statemachine_change_state(ctx, AUTHENTICATING);
      return;
    }

  usleep(500000);
}

/******************************************************************
 *
 * Switch to authenticating state.
 *
 ******************************************************************/
int statemachine_change_to_authenticating(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  // We can only reach authenticating state from CONNECTING or RESTART
  // states.
  if ((ctx->statemachine->curState != CONNECTING) &&
      (ctx->statemachine->curState != RESTART))
    {
      debug_printf(DEBUG_NORMAL, "Attempt to change to AUTHENTICATING state "
		   "from an invalid state!\n");
      return statemachine_change_state(ctx, DISCONNECTED);
    }

  ctx->statemachine->startCount = 0;
  ctx->statemachine->suppSuccess = FALSE;
  ctx->statemachine->suppFail = FALSE;
  ctx->statemachine->suppTimeout = FALSE;
  ctx->statemachine->keyRun = FALSE;
  ctx->statemachine->keyDone = FALSE;
  ctx->statemachine->suppStart = TRUE;

  ctx->statemachine->curState = AUTHENTICATING;
  xsup_ipc_send_auth_state(ctx);

  return XENONE;
}

/******************************************************************
 *
 * Process authenticating state.
 *
 ******************************************************************/
void statemachine_do_authenticating(struct interface_data *ctx)
{
  if (ctx->statemachine->suppSuccess &&
      ctx->statemachine->portValid)
    {
      statemachine_change_state(ctx, AUTHENTICATED);
      return;
    }
  
  if (ctx->statemachine->suppFail || (ctx->statemachine->keyDone &&
				      !ctx->statemachine->portValid))
    {
      statemachine_change_state(ctx, HELD);
      return;
    }
  
  if (ctx->statemachine->suppTimeout)
    {
      statemachine_change_state(ctx, CONNECTING);
      return;
    }
}

/******************************************************************
 *
 * Change to authenticated state.
 *
 ******************************************************************/
int statemachine_change_to_authenticated(struct interface_data *ctx)
{
  struct config_globals *globals;

  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }
  
  ctx->statemachine->suppPortStatus = AUTHORIZED;

  ctx->statemachine->curState = AUTHENTICATED;
  xsup_ipc_send_auth_state(ctx);

  // Since we changed to authenticated state, we want to start the passive
  // scan timer, if the user hasn't disabled it.
  globals = config_get_globals();

  if ((!globals) || (TEST_FLAG(globals->flags, CONFIG_GLOBALS_PASSIVE_SCAN)))
    {
      // Start scan timer.
      if (timer_check_existing(PASSIVE_SCAN_TIMER) == FALSE)
	{
	  // Set up a new timer, since this is the first time we have set a timer.
	  debug_printf(DEBUG_EVERYTHING, "Starting new passive scan timer.\n");
	  timer_add_timer(PASSIVE_SCAN_TIMER, globals->passive_timeout, NULL,
			  cardif_passive_scan_timeout);
	} 
      else
	{
	  // Reset the timer so we don't scan sooner than needed.
	  debug_printf(DEBUG_EVERYTHING, "Resetting passive scan timer.\n");
	  timer_reset_timer_count(PASSIVE_SCAN_TIMER, 
				  globals->passive_timeout);
	}
    }

  return XENONE;
}

/******************************************************************
 *
 * Process authenticated state.
 *
 ******************************************************************/
void statemachine_do_authenticated(struct interface_data *ctx)
{
  if (ctx->statemachine->eapolEap && 
      ctx->statemachine->portValid)
    {
      statemachine_change_state(ctx, RESTART);
      return;
    }
  
  if (!ctx->statemachine->portValid)
    {
      statemachine_change_state(ctx, DISCONNECTED);
      return;
    }

  usleep(500000);
}

/******************************************************************
 *
 * Change from whatever state we are currently in, to a new state.
 *
 ******************************************************************/
int statemachine_change_state(struct interface_data *ctx, int newstate)
{
  int retval = XENONE;

  debug_printf(DEBUG_STATE, "Changing from ");
  statemachine_disp_state(DEBUG_STATE, ctx->statemachine->curState);
  debug_printf_nl(DEBUG_STATE, " to ");
  statemachine_disp_state(DEBUG_STATE, newstate);
  debug_printf_nl(DEBUG_STATE, ".\n");

  switch (newstate)
    {
    case LOGOFF:
      retval=statemachine_change_to_logoff(ctx);
      break;

    case DISCONNECTED:
      retval=statemachine_change_to_disconnected(ctx);
      break;

    case CONNECTING:
      retval=statemachine_change_to_connecting(ctx);
      break;

    case ACQUIRED:
      debug_printf(DEBUG_NORMAL, "Attempt to change to ACQUIRED state. "
		   "WTF are you doing!?\n");
      retval=statemachine_change_to_disconnected(ctx);
      break;

    case AUTHENTICATING:
      retval=statemachine_change_to_authenticating(ctx);
      break;

    case HELD:
      retval=statemachine_change_to_held(ctx);
      break;

    case AUTHENTICATED:
      retval=statemachine_change_to_authenticated(ctx);
      break;

    case RESTART:
      retval=statemachine_change_to_restart(ctx);
      break;

    case S_FORCE_AUTH:
      retval=statemachine_change_to_s_force_auth(ctx);
      break;

    case S_FORCE_UNAUTH:
      retval=statemachine_change_to_s_force_unauth(ctx);
      break;
    }

  return retval;
}

/******************************************
 *
 * Check for cases where we should change state no matter what state we
 * are currently in.
 *
 ******************************************/
void statemachine_check_global_transition(struct interface_data *ctx)
{
  if ((ctx->statemachine->userLogoff && !ctx->statemachine->logoffSent)
      && !(ctx->statemachine->initialize || 
	   !ctx->statemachine->portEnabled))
    {
      statemachine_change_state(ctx, LOGOFF);
    }

  if ((ctx->statemachine->portControl == FORCE_AUTHORIZED) &&
      (ctx->statemachine->sPortMode != ctx->statemachine->portControl)
      && !(ctx->statemachine->initialize || 
	   !ctx->statemachine->portEnabled))
    {
      statemachine_change_state(ctx, S_FORCE_AUTH);
    }

  if ((ctx->statemachine->portControl == FORCE_UNAUTHORIZED) &&
      (ctx->statemachine->sPortMode != ctx->statemachine->portControl)
      && !(ctx->statemachine->initialize ||
	   !ctx->statemachine->portEnabled))
    {
      statemachine_change_state(ctx, S_FORCE_UNAUTH);
    }

  if ((ctx->statemachine->portControl == AUTO) &&
      ((ctx->statemachine->sPortMode != ctx->statemachine->portControl)
      || ctx->statemachine->initialize || 
       !ctx->statemachine->portEnabled))
    {
      statemachine_change_state(ctx, DISCONNECTED);
    }
}

/******************************************
 * 
 * Update all state machine timers.
 *
 ******************************************/
void statemachine_timer_tick(struct interface_data *ctx)
{
  // The clock ticked -- Update all of the needed counters.
  dec_if_nz(&ctx->statemachine->authWhile);
  dec_if_nz(&ctx->statemachine->heldWhile);
  dec_if_nz(&ctx->statemachine->startWhen);
  
  ctx->statemachine->tick = FALSE;
  debug_printf(DEBUG_EVERYTHING, "Clock tick! authWhile=%d heldWhile=%d "
	       "startWhen=%d curState=",
	       ctx->statemachine->authWhile,
	       ctx->statemachine->heldWhile,
	       ctx->statemachine->startWhen);

  statemachine_disp_state(DEBUG_EVERYTHING, ctx->statemachine->curState);
  debug_printf_nl(DEBUG_EVERYTHING, "\n");
}

/******************************************
 *
 * Process the state machine, send a frame if we need to.  Returns >0 if
 * there is a frame to be sent.
 *
 ******************************************/
int statemachine_run(struct interface_data *thisint)
{
  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed in to "
		   "%s:%d!\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (!thisint->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Statemachine is not set up correctly in "
		   "%s:%d!\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (thisint->statemachine->tick == TRUE)
    {
      statemachine_timer_tick(thisint);

      statemachine_check_global_transition(thisint);
    }

  thisint->statemachine->portEnabled = cardif_get_if_state(thisint);

  if (thisint->statemachine->portEnabled == FALSE)
    {
      // Change our wireless state to port down state.
      //      wireless_sm_change_state(PORT_DOWN, thisint);
      sleep(1);
      // And bail out.
      return XENONE;
    }

  switch(thisint->statemachine->curState)
    {
    case S_FORCE_AUTH:
      statemachine_do_s_force_auth(thisint);
      break;

    case S_FORCE_UNAUTH:
      statemachine_do_s_force_unauth(thisint);
      break;

    case LOGOFF:
      statemachine_do_logoff(thisint);
      break;

    case DISCONNECTED:
      statemachine_do_disconnected(thisint);
      break;

    case HELD:
      statemachine_do_held(thisint);
      break;

    case CONNECTING:
      statemachine_do_connecting(thisint);
      break;

    case RESTART:
      statemachine_do_restart(thisint);
      break;

    case AUTHENTICATING:
      statemachine_do_authenticating(thisint);
      break;

    case AUTHENTICATED:
      statemachine_do_authenticated(thisint);
      break;
    }

  return backend_sm_run(thisint);
}

/*****************************************
 *
 * Clean up our state machine.
 *
 *****************************************/
int statemachine_cleanup(struct interface_data *thisint)
{
  debug_printf(DEBUG_EVERYTHING, "Doing statemachine cleanup!\n");

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to statemachine_cleanup()!\n");
      return XEMALLOC;
    }

  if (thisint->statemachine != NULL)
    {
      free(thisint->statemachine);
      thisint->statemachine = NULL;
    }

  backend_sm_deinit(thisint);
  
  return XENONE;
}



/*****************************************
 *
 * Create a logoff frame to be sent out to the network.
 *
 *****************************************/
int txLogoff(struct interface_data *thisint)
{
  int eapolver;
  struct config_network *network_data;

  // Pick up our current network configuration.
  network_data = config_get_network_config();

  if (network_data == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "No network information available.  Not "
		   "sending a logoff!\n");
      return XENONE;      // Not having data here is NOT an error!
    }

  // If we aren't using 802.1X don't send logoffs.
  if (network_data->methods == NULL) return XENONE;
  if (network_data->methods->method_num == STATIC_WEP_METHOD) return XENONE;

  // If we are using WPA-PSK, don't send logoffs.
  if (network_data->methods->method_num == WPA_PSK) return XENONE;

  debug_printf(DEBUG_STATE, "Sending EAPOL-Logoff Frame.\n");

  eapolver = eapol_get_eapol_ver();

  eapol_build_header(EAPOL_LOGOFF, 0, eapolver, (char *) thisint->sendframe);
  thisint->send_size = OFFSET_TO_EAP;

  snmp_dot1xSuppEapolLogoffFramesTx();

  cardif_sendframe(thisint);

  return XENONE;
}


/*********************************************
 *
 * Build an EAPoL Start frame to be sent out to the network.
 *
 *********************************************/
int txStart(struct interface_data *thisint)
{
  int eapolver;
  struct config_network *network_data;

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  network_data = config_get_network_config();

  if ((!network_data) || (!network_data->methods))
    {
      // It is possible that we have an empty network clause, in which case
      // we want to skip over doing a Start, since it is not needed.
      debug_printf(DEBUG_CONFIG, "Invalid network configuration data! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      debug_printf(DEBUG_CONFIG, "Depending on the specific network clause,"
		   " this may not be anything to worry about!\n");
      return XEMALLOC;
    }

  // If we are using WPA-PSK, don't send starts.
  if (network_data->methods->method_num == WPA_PSK) return XENONE;

  debug_printf(DEBUG_STATE, "Sending EAPOL-Start Frame.\n");

  eapolver = eapol_get_eapol_ver();
  
  eapol_build_header(EAPOL_START, 0, network_data->force_eapol_ver,
		     (char *) thisint->sendframe);
  thisint->send_size = OFFSET_TO_EAP;

  snmp_dot1xSuppEapolStartFramesTx();

  cardif_sendframe(thisint);

  return XENONE; 
}
