package format

import (
	"bytes"
	"encoding/hex"
	"github.com/DCSO/balboa/observation"
	"github.com/farsightsec/go-nmsg"
	"github.com/sirupsen/logrus/hooks/test"
	"testing"
	"time"
)

const nmsg_ok = "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"

func TestNmsgFormatFail(t *testing.T) {
	hook := test.NewGlobal()

	resultObs := make([]observation.InputObservation, 0)
	resChan := make(chan observation.InputObservation)
	go func() {
		for o := range resChan {
			resultObs = append(resultObs, o)
		}
	}()

	stopChan := make(chan bool)

	err := MakePacketbeatInputObservations([]byte(`babanana`), "foo", resChan, stopChan)
	if err != nil {
		t.Fatal(err)
	}

	close(resChan)
	close(stopChan)

	if len(resultObs) != 0 {
		t.Fail()
	}
	if len(hook.Entries) != 1 {
		t.Fail()
	}
}

func TestNmsgFormatEmpty(t *testing.T) {
	hook := test.NewGlobal()

	resultObs := make([]observation.InputObservation, 0)
	resChan := make(chan observation.InputObservation)
	go func() {
		for o := range resChan {
			resultObs = append(resultObs, o)
		}
	}()

	stopChan := make(chan bool)

	err := MakePacketbeatInputObservations([]byte(""), "foo", resChan, stopChan)
	if err != nil {
		t.Fatal(err)
	}

	close(resChan)
	close(stopChan)

	if len(resultObs) != 0 {
		t.Fail()
	}
	if len(hook.Entries) != 1 {
		t.Fail()
	}
}

func TestNmsgFormat(t *testing.T) {
	hook := test.NewGlobal()

	resultObs := make([]observation.InputObservation, 0)
	resChan := make(chan observation.InputObservation)
	go func() {
		for o := range resChan {
			resultObs = append(resultObs, o)
		}
	}()

	stopChan := make(chan bool)

	dst := make([]byte, hex.DecodedLen(len(nmsg_ok)))
	n, err := hex.Decode(dst, []byte(nmsg_ok))
	if err != nil {
		t.Fatal(err)
	}

	r := bytes.NewReader(dst[:n])
	input := nmsg.NewInput(r, 1480)
	for {
		payload, err := input.Recv()
		if err != nil {
			break
		}
		err = MakeNmsgInputObservations(payload.GetPayload(), "foo", resChan, stopChan)
		if err != nil {
			t.Fatal(err)
		}
	}

	time.Sleep(500 * time.Millisecond)
	close(resChan)
	close(stopChan)

	if len(resultObs) != 100 {
		t.Fail()
	}
	if len(hook.Entries) != 0 {
		t.Fail()
	}
}
